package rip.marie.util.wrapper;

import lombok.AllArgsConstructor;
import lombok.Getter;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.tree.ClassNode;
import org.objectweb.asm.tree.FieldNode;
import org.objectweb.asm.tree.InsnNode;
import org.objectweb.asm.tree.MethodNode;
import rip.marie.obfuscator.ZywcfuscatorBase;

import java.lang.reflect.Modifier;
import java.util.ArrayList;

/**
 * Wraps around ClassNode to keep util calls simple
 *
 * @author Marie
 */
@Getter
@AllArgsConstructor
public final class ClassWrapper {

    private final ArrayList<MethodWrapper> methods = new ArrayList<>();
    private final ArrayList<FieldWrapper> fields = new ArrayList<>();
    private final ClassNode base;

    public void pre() {

        for (MethodNode method : base.methods) {

            methods.add(new MethodWrapper(this, method));

        }

        for (FieldNode field : base.fields) {

            fields.add(new FieldWrapper(this, field));

        }

    }

    public void post() {

        base.methods.clear();

        for (MethodWrapper method : methods) {
            base.methods.add(method.getBase());
        }

        methods.clear();

        base.fields.clear();

        for (FieldWrapper field : fields) {
            base.fields.add(field.getBase());
        }

        fields.clear();

    }

    public boolean isModifiable(ZywcfuscatorBase base) {
        for (String blacklistedPath : base.getBlacklistedPaths()) {
            if (this.base.name.startsWith(blacklistedPath)) return false;
        }

        boolean foundMatch = false;
        for (String whitelistedPath : base.getWhitelistedPaths()) {
            if (this.base.name.startsWith(whitelistedPath)) {
                foundMatch = true;
                break;
            }
        }

        if (!foundMatch) {
            return false;
        }

        return !(Modifier.isAbstract(this.base.access) || Modifier.isInterface(this.base.access));
    }

    public MethodWrapper getStaticInit() {
        MethodWrapper methodWrapper = methods.stream()
                .filter(method -> method.getBase().name.equals("<clinit>"))
                .findAny().orElse(null);

        if (methodWrapper == null) {
            MethodNode clinit = new MethodNode(Opcodes.ACC_STATIC, "<clinit>", "()V", null, null);

            clinit.instructions.add(new InsnNode(Opcodes.RETURN));

            methods.add(new MethodWrapper(this, clinit));

            return getStaticInit();
        }

        return methodWrapper;
    }

    public String getDottedName() {
        return base.name.replace("/", ".");
    }
}
