package rip.marie.util;

import lombok.experimental.UtilityClass;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.tree.ClassNode;
import rip.marie.logger.LogLevel;
import rip.marie.logger.Logger;
import rip.marie.obfuscator.ZywcfuscatorBase;
import rip.marie.util.wrapper.BinaryFileWrapper;
import rip.marie.util.wrapper.ClassWrapper;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

/**
 * For writing and loading classes
 *
 * @author Marie
 */
@UtilityClass
public final class FileUtil {

    /**
     * Loads a JAR file specified in the Obfuscators input path, reads the classes,
     * and wraps them into ClassWrapper objects for further processing.
     *
     * @param base Zywcfuscator instance
     */
    public static void loadJar(ZywcfuscatorBase base) {
        Path inputPath = base.getInput();

        if (inputPath == null || !Files.exists(inputPath)) {
            throw new IllegalArgumentException("Invalid input path: " + inputPath);
        }

        try (InputStream inputFileStream = Files.newInputStream(inputPath); ZipInputStream zipStream = new ZipInputStream(inputFileStream)) {
            ZipEntry zipEntry;
            while ((zipEntry = zipStream.getNextEntry()) != null) {
                if (zipEntry.getName().endsWith(".class")) {
                    try {
                        ClassReader reader = new ClassReader(zipStream);

                        ClassNode node = new ClassNode();

                        reader.accept(node, base.getClassReaderFlags());

                        base.getClasses().add(new ClassWrapper(node));
                    } catch (IOException e) {
                        Logger.log(LogLevel.WARN, "Failed to load class file \"", zipEntry.getName(), "\"!");
                    }
                } else {
                    base.getBinaryFiles().add(new BinaryFileWrapper(zipEntry.getName(), zipStream.readAllBytes()));
                }
                zipStream.closeEntry();
            }
        } catch (IOException e) {
            throw new RuntimeException("Failed to load JAR file: " + inputPath, e);
        }
    }

    /**
     * Saves the obfuscated classes to the output JAR file specified in the Obfuscator
     *
     * @param base The Obfuscator instance
     */
    public static void saveJar(ZywcfuscatorBase base) {
        Path outputPath = base.getOutput();

        if (outputPath == null) {
            throw new IllegalArgumentException("Output path is not set.");
        }

        try (OutputStream fileStream = Files.newOutputStream(outputPath);
             ZipOutputStream zipOutputStream = new ZipOutputStream(fileStream)) {

            for (ClassWrapper classWrapper : base.getClasses()) {
                ClassWriter writer = new ClassWriter(base.getWhitelistedClasses().contains(classWrapper) ? base.getClassWriterFlags() : 0);

                Logger.log(LogLevel.DEBUG, classWrapper.getBase().name);
                classWrapper.getBase().accept(writer);

                String entryName = classWrapper.getBase().name + ".class";

                zipOutputStream.putNextEntry(new ZipEntry(entryName));
                zipOutputStream.write(writer.toByteArray());
                zipOutputStream.closeEntry();
            }

            for (BinaryFileWrapper ncf : base.getBinaryFiles()) {
                if (ncf.getFileName().endsWith("/")) continue;

                zipOutputStream.putNextEntry(new ZipEntry(ncf.getFileName()));
                zipOutputStream.write(ncf.getFileData());
                zipOutputStream.closeEntry();
            }

        } catch (IOException e) {
            throw new RuntimeException("Failed to save JAR file: " + outputPath, e);
        }
    }

}
