package rip.marie.deadCodeEngine;

import org.objectweb.asm.tree.*;
import rip.marie.logger.LogLevel;
import rip.marie.logger.Logger;
import rip.marie.util.asm.BytecodeUtil;
import rip.marie.util.wrapper.InsnListWrapper;
import rip.marie.util.wrapper.MethodWrapper;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Random;

public class DeadCodeGenerator extends BytecodeUtil {

    private final ArrayList<LocalVariableObject> lvos = new ArrayList<>();
    private final Random r = new Random();
    private final MethodWrapper mw;

    public DeadCodeGenerator(MethodWrapper mw) {
        this.mw = mw;

        this.mw.getInstructions().forEach(ain -> {

            if (ain instanceof VarInsnNode vin) {
                if (!Modifier.isStatic(mw.getBase().access) && vin.var == 0) return;
                if (lvos.stream().anyMatch(lvo -> lvo.idx() == vin.var)) return;

                switch (vin.getOpcode()) {

                    case ISTORE, ILOAD -> lvos.add(new LocalVariableObject(vin.var, LocalVariableObjectType.INT));
                    case LSTORE, LLOAD -> lvos.add(new LocalVariableObject(vin.var, LocalVariableObjectType.LONG));
                    case DSTORE, DLOAD -> lvos.add(new LocalVariableObject(vin.var, LocalVariableObjectType.DOUBLE));
                    case FSTORE, FLOAD -> lvos.add(new LocalVariableObject(vin.var, LocalVariableObjectType.FLOAT));
                    case ASTORE, ALOAD -> lvos.add(new LocalVariableObject(vin.var, LocalVariableObjectType.OBJECT));
                    default -> throw new IllegalStateException("Wtf");

                }

            }

        });
    }

    public void insertDeadCode(AbstractInsnNode insn) {

        InsnListWrapper ilw = new InsnListWrapper();

        LabelNode handler = new LabelNode();

        int i1 = r.nextInt();

        int i2 = Integer.MIN_VALUE;

        int op = Integer.MIN_VALUE;

        switch (r.nextInt(6)) {
            case 0:
                i2 = i1;

                op = IF_ICMPEQ;
                break;
            case 1:
                do {
                    i2 = r.nextInt();
                } while (i1 == i2);

                op = IF_ICMPNE;
                break;
            case 2:
                do {
                    i2 = r.nextInt();
                } while (i1 >= i2);

                op = IF_ICMPLE;
                break;
            case 3:
                do {
                    i2 = r.nextInt();
                } while (i1 <= i2);

                op = IF_ICMPGE;
                break;
            case 4:
                do {
                    i2 = r.nextInt();
                } while (i1 < i2);

                op = IF_ICMPGT;
                break;
            case 5:
                do {
                    i2 = r.nextInt();
                } while (i1 > i2);

                op = IF_ICMPLT;
                break;
        }

        Collections.shuffle(lvos);
        InsnListWrapper insnList = actualDeadCode();

        if (insnList.size() != 0) {
            ilw.add(makeInteger(i1));
            ilw.add(makeInteger(i2));
            ilw.add(new JumpInsnNode(op, handler));
            ilw.add(insnList);
            ilw.add(handler);

            mw.getInstructions().insert(insn, ilw);
        }

    }

    private InsnListWrapper actualDeadCode() {

        InsnListWrapper ilw = new InsnListWrapper();

        if (lvos.isEmpty()) {
            Logger.log(LogLevel.DEBUG, "Did not find and LVOs inside method ", mw.getBase().name, mw.getBase().desc);
        }

        lvos.stream().filter(_ -> r.nextBoolean()).forEach(lvo -> {

            switch (lvo.type()) {

                case INT -> {
                    if (r.nextBoolean()) {
                        ilw.add(makeInteger(r.nextInt()));
                        ilw.add(new VarInsnNode(ISTORE, lvo.idx()));
                    }
                }

                case LONG -> {
                    if (r.nextBoolean()) {
                        ilw.add(makeInteger(r.nextInt()));
                        ilw.add2(I2L);
                        ilw.add(new VarInsnNode(LSTORE, lvo.idx()));
                    } else {
                        ilw.add(new LdcInsnNode(Long.valueOf(r.nextLong())));
                        ilw.add(new VarInsnNode(LSTORE, lvo.idx()));
                    }
                }

                case DOUBLE -> {
                    ilw.add(new LdcInsnNode(r.nextGaussian() * Math.random()));
                    ilw.add(new VarInsnNode(DSTORE, lvo.idx()));
                }

                case FLOAT -> {
                    ilw.add(new LdcInsnNode(r.nextFloat() * r.nextFloat()));
                    ilw.add(new VarInsnNode(FSTORE, lvo.idx()));
                }

                case OBJECT -> {
                    ilw.add2(ACONST_NULL);
                    ilw.add(new VarInsnNode(ASTORE, lvo.idx()));
                }

            }

        });

        return ilw;

    }

}
