package org.volt.impl.event;

import it.unimi.dsi.fastutil.objects.*;
import org.volt.api.event.EventReceiver;
import org.volt.api.event.IEvent;
import org.volt.api.event.IEventBus;

import java.lang.reflect.Method;

/**
 * Quick and easy implementation
 * @author marie
 */
public final class EventBus implements IEventBus
{
    /**
     * Holds a list of methods to call for each event
     */
    private final Object2ReferenceMap<Class<?>, ObjectArrayList<Method>> eventToMethodsMap = new Object2ReferenceOpenHashMap<>();

    /**
     * Holds a parent class instance for each method
     * TODO: Optimize this since it might become a bit large
     */
    private final Object2ReferenceMap<Method, Object> methodToInstanceMap = new Object2ReferenceOpenHashMap<>();

    @Override
    public <T> void call(final T event)
    {
        try
        {
            final Class<?> clazz = event.getClass();

            if (!eventToMethodsMap.containsKey(clazz)) return;

            for (final Method method : eventToMethodsMap.get(clazz))
            {
                method.invoke(methodToInstanceMap.get(method), event);
            }
        }
        catch (final Throwable t)
        {
            throw new RuntimeException(t);
        }
    }

    @Override
    public void register(final Object o)
    {
        final Class<?> clazz = o.getClass();

        for (final Method method : clazz.getMethods())
        {
            // fully exit the register method since the class has already been registered
            if (methodToInstanceMap.containsKey(method)) return;

            if (!method.isAnnotationPresent(EventReceiver.class)) continue;

            final Class<?>[] parameterTypes = method.getParameterTypes();

            if (parameterTypes.length != 1) continue;

            final Class<?> parameterType = parameterTypes[0];

            if (!isEvent(parameterType)) continue;

            getList(parameterType).add(method);
            methodToInstanceMap.put(method, o);
        }
    }

    @Override
    public void unregister(final Object o)
    {
        final Class<?> clazz = o.getClass();

        for (final Method method : clazz.getMethods())
        {
            for (final ObjectArrayList<Method> value : eventToMethodsMap.values())
            {
                value.remove(method);
            }

            methodToInstanceMap.remove(method);
        }
    }

    /**
     * Very quick and simple check if the class has IEvent as an interface
     * @param clazz Class to check
     * @return If one of the interfaces of the input class is IEvent
     * @see IEvent
     */
    private boolean isEvent(final Class<?> clazz)
    {
        for (final Class<?> i : clazz.getInterfaces())
        {
            if (i == IEvent.class)
            {
                return true;
            }
        }

        return false;
    }

    /**
     * Gets an arraylist from the event to method arraylist map, and if the arraylist does not exist, create it
     * @param clazz Event to method arraylist map key
     * @return Arraylist from the event to method arraylist map
     */
    private ObjectArrayList<Method> getList(final Class<?> clazz)
    {
        if (eventToMethodsMap.containsKey(clazz))
        {
            return eventToMethodsMap.get(clazz);
        }
        else
        {
            final ObjectArrayList<Method> objects = new ObjectArrayList<>();

            eventToMethodsMap.put(clazz, objects);
            return objects;
        }
    }
}
