package net.minecraft.client.audio;

import com.google.common.collect.Lists;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Map.Entry;
import net.minecraft.client.resources.IResource;
import net.minecraft.client.resources.IResourceManager;
import net.minecraft.client.resources.IResourceManagerReloadListener;
import net.minecraft.client.settings.GameSettings;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.util.ITickable;
import net.minecraft.util.ResourceLocation;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class SoundHandler implements IResourceManagerReloadListener, ITickable {
    private static final Logger logger = LogManager.getLogger();
    private static final Gson GSON = new GsonBuilder().registerTypeAdapter(SoundList.class, new SoundListSerializer()).create();
    private static final ParameterizedType TYPE = new ParameterizedType() {
        public Type[] getActualTypeArguments() { return new Type[] {String.class, SoundList.class}; }
        public Type getRawType() { return Map.class; }
        public Type getOwnerType() { return null; }
    };
    public static final SoundPoolEntry missing_sound = new SoundPoolEntry(new ResourceLocation("meta:missing_sound"), 0.0D, 0.0D, false);
    private final SoundRegistry sndRegistry = new SoundRegistry();
    private final SoundManager sndManager;
    private final IResourceManager mcResourceManager;

    public SoundHandler(IResourceManager manager, GameSettings gameSettingsIn) {
        this.mcResourceManager = manager;
        this.sndManager = new SoundManager(this, gameSettingsIn);
    }

    public void onResourceManagerReload(IResourceManager resourceManager) {
        this.sndRegistry.clearMap();
        for (String s : resourceManager.getResourceDomains()) {
            try {
                for (IResource iresource : resourceManager.getAllResources(new ResourceLocation(s, "sounds.json"))) {
                    try {
                        Map<String, SoundList> map = this.getSoundMap(iresource.getInputStream());
                        for (Entry<String, SoundList> entry : map.entrySet()) {
                            this.loadSoundResource(new ResourceLocation(s, entry.getKey()), entry.getValue());
                        }
                    } catch (RuntimeException e) {
                        logger.warn("Invalid sounds.json in domain {}", s, e);
                    }
                }
            } catch (IOException e) {
                logger.warn("Failed to load sounds.json for domain {}", s, e);
            }
        }
        this.sndManager.reloadSoundSystem();
    }

    protected Map<String, SoundList> getSoundMap(InputStream stream) {
        if (stream == null) return new java.util.HashMap<>();
        try (InputStreamReader reader = new InputStreamReader(stream)) {
            return GSON.fromJson(reader, TYPE);
        } catch (IOException e) {
            logger.warn("Failed to parse sounds.json", e);
            return new java.util.HashMap<>();
        }
    }

    private void loadSoundResource(ResourceLocation location, SoundList sounds) {
        if (location == null || sounds == null) return;
        boolean flag = !this.sndRegistry.containsKey(location);
        SoundEventAccessorComposite soundeventaccessorcomposite;
        if (!flag && !sounds.canReplaceExisting()) {
            soundeventaccessorcomposite = this.sndRegistry.getObject(location);
        } else {
            soundeventaccessorcomposite = new SoundEventAccessorComposite(location, 1.0D, 1.0D, sounds.getSoundCategory());
            this.sndRegistry.registerSound(soundeventaccessorcomposite);
        }
        for (SoundList.SoundEntry soundentry : sounds.getSoundList()) {
            if (soundentry == null) continue;
            String s = soundentry.getSoundEntryName();
            ResourceLocation resourcelocation = new ResourceLocation(s);
            String s1 = s.contains(":") ? resourcelocation.getResourceDomain() : location.getResourceDomain();
            ISoundEventAccessor<SoundPoolEntry> isoundeventaccessor;
            switch (soundentry.getSoundEntryType()) {
                case FILE:
                    ResourceLocation resourcelocation1 = new ResourceLocation(s1, "sounds/" + resourcelocation.getResourcePath() + ".ogg");
                    try {
                        this.mcResourceManager.getResource(resourcelocation1);
                        isoundeventaccessor = new SoundEventAccessor(new SoundPoolEntry(resourcelocation1, soundentry.getSoundEntryPitch(), soundentry.getSoundEntryVolume(), soundentry.isStreaming()), soundentry.getSoundEntryWeight());
                    } catch (IOException e) {
                        logger.warn("Failed to load sound file {} for event {}", resourcelocation1, location, e);
                        continue;
                    }
                    break;
                case SOUND_EVENT:
                    ResourceLocation eventLocation = new ResourceLocation(s1, soundentry.getSoundEntryName());
                    isoundeventaccessor = new ISoundEventAccessor<SoundPoolEntry>() {
                        public int getWeight() {
                            SoundEventAccessorComposite composite = SoundHandler.this.sndRegistry.getObject(eventLocation);
                            return composite == null ? 0 : composite.getWeight();
                        }
                        public SoundPoolEntry cloneEntry() {
                            SoundEventAccessorComposite composite = SoundHandler.this.sndRegistry.getObject(eventLocation);
                            return composite == null ? SoundHandler.missing_sound : composite.cloneEntry();
                        }
                    };
                    break;
                default:
                    logger.error("Unknown sound entry type for {}", s);
                    continue;
            }
            soundeventaccessorcomposite.addSoundToEventPool(isoundeventaccessor);
        }
    }

    public SoundEventAccessorComposite getSound(ResourceLocation location) {
        return location == null ? null : this.sndRegistry.getObject(location);
    }

    public void playSoundFromFullPath(String fullPath, float volume, float pitch) {
        if (fullPath == null) return;
        try (InputStream inputStream = getClass().getResourceAsStream(fullPath)) {
            if (inputStream == null) {
                logger.warn("Sound resource not found at path: {}", fullPath);
                return;
            }
            sndManager.playSoundFromStream(inputStream, Math.max(0.0f, Math.min(volume, 1.0f)), Math.max(0.5f, Math.min(pitch, 2.0f)), fullPath);
        } catch (IOException e) {
            logger.error("Failed to load sound from path: {}", fullPath, e);
        }
    }

    public void playSound(ISound sound) {
        if (sound != null) this.sndManager.playSound(sound);
    }

    public void playDelayedSound(ISound sound, int delay) {
        if (sound != null) this.sndManager.playDelayedSound(sound, delay);
    }

    public void setListener(EntityPlayer player, float renderPartialTicks) {
        this.sndManager.setListener(player, renderPartialTicks);
    }

    public void pauseSounds() {
        this.sndManager.pauseAllSounds();
    }

    public void stopSounds() {
        this.sndManager.stopAllSounds();
    }

    public void unloadSounds() {
        this.sndManager.unloadSoundSystem();
    }

    public void update() {
        this.sndManager.updateAllSounds();
    }

    public void resumeSounds() {
        this.sndManager.resumeAllSounds();
    }

    public void setSoundLevel(SoundCategory category, float volume) {
        if (category != null) {
            if (category == SoundCategory.MASTER && volume <= 0.0F) this.stopSounds();
            this.sndManager.setSoundCategoryVolume(category, Math.max(0.0f, Math.min(volume, 1.0f)));
        }
    }

    public void stopSound(ISound sound) {
        if (sound != null) this.sndManager.stopSound(sound);
    }

    public SoundEventAccessorComposite getRandomSoundFromCategories(SoundCategory... categories) {
        if (categories == null || categories.length == 0) return null;
        List<SoundEventAccessorComposite> list = Lists.newArrayList();
        for (ResourceLocation resourcelocation : this.sndRegistry.getKeys()) {
            SoundEventAccessorComposite soundeventaccessorcomposite = this.sndRegistry.getObject(resourcelocation);
            if (soundeventaccessorcomposite != null && ArrayUtils.contains(categories, soundeventaccessorcomposite.getSoundCategory())) list.add(soundeventaccessorcomposite);
        }
        return list.isEmpty() ? null : list.get(new Random().nextInt(list.size()));
    }

    public boolean isSoundPlaying(ISound sound) {
        return sound != null && this.sndManager.isSoundPlaying(sound);
    }
}