package war.jnt.exhaust.compiler;

import lombok.SneakyThrows;
import war.configuration.ConfigurationSection;
import war.jnt.cache.Cache;
import war.jnt.core.Processor;
import war.jnt.core.header.Header;
import war.jnt.core.source.Source;
import war.jnt.dash.Ansi;
import war.jnt.dash.Level;
import war.jnt.dash.Logger;
import war.jnt.dash.Origin;
import war.jnt.utility.timing.Timing;

import java.io.*;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.zip.Deflater;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import static war.jnt.dash.Ansi.Color.*;

public class CompilerZig {

    private static final Logger logger = Logger.INSTANCE;
    private static final Timing timing = new Timing();
    private final Processor processor;

    public CompilerZig(Processor processor) {
        this.processor = processor;
    }

    public File downloadZig(ConfigurationSection config) {
        logger.logln(Level.INFO, Origin.EXHAUST, "Searching for zig compiler...");

        String extension = System.getProperty("os.name").toLowerCase().contains("windows") ? ".exe" : "";

        File root = new File(config.getString("zig.installation"));
        Queue<File> files = new LinkedList<>();
        files.add(root);
        while (!files.isEmpty()) {
            File file = files.poll();
            if (file.isDirectory()) {
                for (File f : Objects.requireNonNull(file.listFiles())) {
                    if (f.isDirectory()) {
                        files.add(f);
                    } else {
                        if (f.getName().equals("zig" + extension)) {
                            return f;
                        }
                    }
                }
            }
        }

        logger.logln(Level.INFO, Origin.EXHAUST, "Zig compiler not found, downloading...");

        String version = config.getString("zig.version");
        String os = config.getString("zig.os");
        String arch = config.getString("zig.arch");
        String zipName = String.format("zig-%s-%s-%s.zip", os, arch, version);
        String urlStr = String.format("https://ziglang.org/download/%s/%s", version, zipName);

        try {
            URL url = URI.create(urlStr).toURL();
            File temp = File.createTempFile("zig", ".zip");
            temp.deleteOnExit();

            URLConnection conn = url.openConnection();
            int totalBytes = conn.getContentLength();
            if (totalBytes <= 0) {
                logger.logln(Level.INFO, Origin.EXHAUST, "Failed to get the size of the file, downloading anyway...");
                totalBytes = -1;
            } else {
                logger.logln(Level.INFO, Origin.EXHAUST, String.format("Downloading zig compiler from %s (%s bytes)", new Ansi().c(WHITE).s(urlStr), new Ansi().c(WHITE).s(String.valueOf(totalBytes))));
            }

            try (InputStream in = conn.getInputStream();
                 FileOutputStream out = new FileOutputStream(temp)) {

                byte[] buffer = new byte[1024];
                long downloaded = 0;
                int r;
                final int BAR_WIDTH = 50;

                while ((r = in.read(buffer)) != -1) {
                    out.write(buffer, 0, r);
                    if (totalBytes > 0) {
                        downloaded += r;
                        int percent = (int) ((downloaded * 100) / totalBytes);

                        int filled = (percent * BAR_WIDTH) / 100;
                        int empty  = BAR_WIDTH - filled;

                        String bar = String.format("[%s%s] %3d%%", "=".repeat(Math.max(0, filled)), " ".repeat(Math.max(0, empty)), percent);

                        logger.rlog(Level.INFO, Origin.EXHAUST, bar);
                    }
                }

                if (totalBytes > 0) {
                    String bar = String.format("[%s] 100%%", "=".repeat(BAR_WIDTH));
                    logger.rlog(Level.INFO, Origin.EXHAUST, bar + "\n");
                } else {
                    logger.logln(Level.INFO, Origin.EXHAUST, "Download complete.");
                }
            }
            logger.logln(Level.INFO, Origin.EXHAUST, String.format("Downloaded zig compiler to %s.", new Ansi().c(WHITE).s(temp.getAbsolutePath())));

            File outputDir = new File(config.getString("zig.installation"));
            if (!outputDir.exists() && !outputDir.mkdirs()) {
                logger.logln(Level.FATAL, Origin.EXHAUST, String.format("Failed to create output directory for zig compiler: %s", new Ansi().c(RED).s(outputDir)));
                return null;
            }

            try (ZipInputStream zis = new ZipInputStream(new FileInputStream(temp))) {
                ZipEntry entry;
                while ((entry = zis.getNextEntry()) != null) {
                    Path outPath = outputDir.toPath().resolve(entry.getName());
                    if (entry.isDirectory()) {
                        Files.createDirectories(outPath);
                    } else {
                        Files.createDirectories(outPath.getParent());
                        try (OutputStream fos = Files.newOutputStream(outPath, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING)) {
                            byte[] buf = new byte[8192];
                            int len;
                            while ((len = zis.read(buf)) > 0) {
                                fos.write(buf, 0, len);
                            }
                        }
                        if (entry.getName().endsWith("zig" + extension)) {
                            if (!outPath.toFile().setExecutable(true)) {
                                logger.logln(Level.FATAL, Origin.EXHAUST, String.format("Failed to set executable permission for zig compiler: %s", new Ansi().c(RED).s(outPath)));
                                return null;
                            }
                        }
                    }
                    zis.closeEntry();
                }
            }

            files.clear();
            files.add(outputDir);
            while (!files.isEmpty()) {
                File dir = files.poll();
                for (File f : Objects.requireNonNull(dir.listFiles())) {
                    if (f.isDirectory()) {
                        files.add(f);
                    } else if (f.getName().equalsIgnoreCase("zig" + extension)) {
                        return f;
                    }
                }
            }

            logger.logln(Level.FATAL, Origin.EXHAUST, "Zig executable not found after extraction in " + outputDir);
            return null;

        } catch (IOException e) {
            logger.logln(Level.FATAL, Origin.EXHAUST, "Failed to download or extract zig compiler: " + e.getMessage());
            return null;
        }
    }

    @SneakyThrows
    public void run(ConfigurationSection config, String dir) {
        timing.begin();

        File zigInstallation = downloadZig(config);

        if (zigInstallation == null || !zigInstallation.exists()) {
            logger.logln(Level.FATAL, Origin.EXHAUST, "Zig compiler not found or failed to download");
            return;
        }

        String holder = String.format("%s/", dir);

        List<String> targets = config.getStringList("targets");
        List<String> debugging = config.getStringList("debug.compilation");

        File buildDir = new File(holder, "build/");

        for (String target : targets) {
            File targetDir = new File(buildDir, target);
            if (targetDir.exists()) {
                Files.walk(targetDir.toPath()).forEach(path -> path.toFile().delete());
            }

            if (!targetDir.exists() && !targetDir.mkdirs()) {
                logger.logln(Level.FATAL, Origin.EXHAUST, String.format("Failed to create target directory: %s", new Ansi().c(RED).s(targetDir)));
                return;
            }

            Set<String> sources = new HashSet<>();
            for (Source source : processor.getSources())
                sources.add(source.getName());
            for (Header header : processor.getHeaders())
                sources.add(header.getName());
            sources.add("lib/intrinsics.c");
            sources.add("lib/intrinsics.h");
            sources.add("lib/jni.h");
            for (String source : sources) {
                File srcFile = new File(holder, source);
                File newSrcFile = new File(targetDir, source);
                if (!newSrcFile.getParentFile().exists()) {
                    if (!newSrcFile.getParentFile().mkdirs()) {
                        logger.logln(Level.FATAL, Origin.EXHAUST, String.format("Failed to create source directory: %s", new Ansi().c(RED).s(newSrcFile.getParentFile())));
                        return;
                    }
                }
                try (InputStream in = new FileInputStream(srcFile);
                     OutputStream out = new FileOutputStream(newSrcFile)) {
                    byte[] buffer = new byte[1024];
                    int len;
                    while ((len = in.read(buffer)) > 0) {
                        out.write(buffer, 0, len);
                    }
                }
            }
        }

        //Stop raping the memory
        for (Source source : processor.getSources()) {
            source.clear();
        }
        System.gc();

        int threads = 4;
        try (ExecutorService executor = Executors.newFixedThreadPool(1)) {
            for (String target : targets) {
                executor.submit(() -> {
                    try {
                        File targetDir = new File(buildDir, target);

                        String targetPath = targetDir.getAbsolutePath();

                        logger.logln(Level.INFO, Origin.EXHAUST, String.format("Linking for %s.", new Ansi().c(WHITE).s(target)));
                        String gcc = zigInstallation.getAbsolutePath() + " cc";

                        List<String> sources = new ArrayList<>();
                        for (Source source : processor.getSources()) {
                            sources.add(source.getName());
                        }
                        sources.add("lib/intrinsics.c");

                        sources.sort(Comparator.comparingLong(src -> {
                            File f = new File(targetPath, src);
                            return -(f.exists() ? f.length() : 0L);
                        }));

                        List<Future<?>> futures = new ArrayList<>();
                        try (ExecutorService ex = Executors.newFixedThreadPool(threads)) {
                            for (String src : sources) {
                                futures.add(ex.submit(() -> {
                                    try {
                                        String srcPath = new File(targetPath, src).getAbsolutePath();
                                        process(String.format("%s -I %s -I %s -I %s %s -c %s -o %s -fPIE -pie", gcc, new File(targetPath, "lib").getAbsolutePath(), new File(targetPath).getAbsolutePath(), new File(targetPath, "classes").getAbsolutePath(), getLinkCommandLine(target), srcPath, srcPath.replaceAll("\\.c$", ".o")), debugging.contains(target));
                                    } catch (Exception e) {
                                        logger.logln(Level.FATAL, Origin.EXHAUST, String.format("Failed to compile source %s for %s.", new Ansi().c(RED).s(src).r(false).c(BRIGHT_RED), new Ansi().c(RED).s(target).r(false).c(BRIGHT_RED)));
                                        e.printStackTrace(System.err);
                                    }
                                }));
                            }
                            for (Future<?> f : futures) {
                                f.get();
                            }
                            ex.shutdown();
                        } catch (Exception e) {
                            logger.logln(Level.FATAL, Origin.EXHAUST, String.format("Failed to compile for %s.", new Ansi().c(RED).s(target).r(false).c(BRIGHT_RED)));
                        }

                        StringBuilder link = new StringBuilder(gcc).append(" -shared ").append(getLinkCommandLine(target));
                        for (String src : sources) {
                            String objPath = new File(targetPath, src.replaceAll("\\.c$", ".o")).getAbsolutePath();
                            link.append(" ").append(objPath);
                        }
                        link.append(" -o ").append(targetPath).append(File.separator).append("out.jnt");

                        process(link.toString(), debugging.contains(target));

                        File out = new File(targetPath + "/out.jnt");
                        File compressed = new File(targetPath + "/out.gz");

                        byte[] bin = Files.readAllBytes(out.toPath());
                        byte[] compressedBin = compress(bin);

                        try (FileOutputStream fos = new FileOutputStream(compressed)) {
                            fos.write(compressedBin);
                        }
                    } catch (Exception e) {
                        logger.logln(Level.FATAL, Origin.EXHAUST, String.format("Failed to compile for %s.", new Ansi().c(RED).s(target).r(false).c(BRIGHT_RED)));
                    }
                });
            }
        } catch (Exception e) {
            processor.clear();
            throw new RuntimeException(e);
        }

        timing.end();
        processor.clear();

        long elapsed = timing.calc();
        logger.logln(Level.INFO, Origin.EXHAUST, String.format("Compiled natives in %s.", new Ansi().c(WHITE).s(String.format("%sms", elapsed))));
    }

    @SneakyThrows
    private void process(String commandLine, boolean debug) {
        Logger.INSTANCE.logln(Level.DEBUG, Origin.EXHAUST, String.format("Running command: %s", new Ansi().c(BRIGHT_CYAN).s(commandLine)));
        StringTokenizer st = new StringTokenizer(commandLine);
        String[] cmdarray = new String[st.countTokens()];
        for (int i = 0; st.hasMoreTokens(); i++)
            cmdarray[i] = st.nextToken();
        ProcessBuilder builder = new ProcessBuilder(cmdarray);

        builder.redirectErrorStream(true);

        Process process = builder.start();

        BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));

        String line;
        while ((line = reader.readLine()) != null) {
            if (debug) {
                logger.logln(Level.FATAL, Origin.EXHAUST, line);
            }
        }

        process.waitFor();
    }

    public String getLinkCommandLine(String target) {
        StringBuilder cmd = new StringBuilder();

        cmd
                .append("-target ").append(target)
//                .append(" -O0 -mcpu=baseline+sse4_1 ")
                .append("-fno-semantic-interposition ")
                .append("-funroll-loops ")
                .append("-finline-functions ")
                .append("-fmerge-all-constants ")
                .append("-fvectorize -fslp-vectorize ")
                .append("-ffp-contract=off -fno-fast-math -fno-math-errno -fno-signed-zeros -fno-trapping-math ")
                .append("-fstrict-overflow -fstrict-aliasing -fomit-frame-pointer ")
                .append("-fstack-protector-strong ")
                .append("-fno-sanitize=undefined ")
                .append("-fPIC -fvisibility=hidden -ffunction-sections -fdata-sections ")
                .append("-Wno-incompatible-pointer-types ")
                .append("-Wignored-optimization-argument ")
                .append("-Wc23-extensions ")
                .append("-D_GLIBCXX_ASSERTIONS -Wformat -Werror=format-security ")
                .append("-Wno-unused-command-line-argument ")
                .append("-Wno-c23-extensions ");

        int classCache = Cache.Companion.cachedClasses();
        int methodCache = Cache.Companion.cachedMethods();
        int fieldCache = Cache.Companion.cachedFields();

        cmd.append("-DCLASS_CACHE=").append(classCache).append(" ")
                .append("-DMETHOD_CACHE=").append(methodCache).append(" ")
                .append("-DFIELD_CACHE=").append(fieldCache).append(" ");

        // Linker flags
        cmd
                .append("-Wl,--sort-section=alignment ")
                .append("-Wl,--discard-all ")
                .append("-Wl,--strip-all ")
                .append("-Wl,--build-id=none ")
                .append("-Wl,--as-needed ")
                .append("-Wl,-Bsymbolic ");
        if (!target.contains("windows")) {
            cmd.append("-Wl,--hash-style=gnu ")
                    .append("-Wl,-z,max-page-size=4096 ")
                    .append("-Wl,-z,relro,-z,now ")
                    .append("-Wl,-z,noexecstack ");
        } else {
            cmd.append("-Wl,--dynamicbase ")
                    .append("-Wl,--nxcompat ");
        }

        return cmd.toString();
    }

    public byte[] compress(byte[] bin) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        Deflater deflater = new Deflater(Deflater.BEST_COMPRESSION, true);
        try (GZIPOutputStream gzip = new GZIPOutputStream(baos) {
            {
                this.def = deflater;
            }
        }) {
            gzip.write(bin);
        }

        return baos.toByteArray();
    }
}
