package war.jar;

import lombok.AllArgsConstructor;
import war.jnt.dash.Ansi;
import war.jnt.dash.Level;
import war.jnt.dash.Logger;
import war.jnt.dash.Origin;
import war.metaphor.tree.JClassNode;

import java.io.*;
import java.nio.file.Path;
import java.util.Set;
import java.util.concurrent.locks.ReentrantLock;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.zip.ZipEntry;

import static war.jnt.dash.Ansi.Color.*;

@AllArgsConstructor
public class JarWriter {

    private Set<JClassNode> classes;
    private Set<JarResource> resources;
    private Manifest manifest;

    public void write(Path in, Path out) {

        try (ByteArrayOutputStream baos = new ByteArrayOutputStream()) {

            try (JarOutputStream jos = new JarOutputStream(baos)) {
                jos.setLevel(9);

                if (manifest != null) {
                    ZipEntry entry = new ZipEntry(JarFile.MANIFEST_NAME);
                    jos.putNextEntry(entry);
                    manifest.write(new BufferedOutputStream(jos));
                    jos.closeEntry();
                }

                ReentrantLock lock = new ReentrantLock();
                classes.forEach(value -> {
                    try {
                        lock.lock();
                        JarEntry entry = new JarEntry(value.name + ".class");
                        try {
                            byte[] bytes = value.compute();
                            jos.putNextEntry(entry);
                            jos.write(bytes);
                            jos.closeEntry();
                        } catch (Exception e) {
                            Logger.INSTANCE.logln(Level.FATAL, Origin.INTAKE, String.format("Failed to write class %s", new Ansi().c(RED).s(value.name)));
                            e.printStackTrace(System.err);
                        } finally {
                            lock.unlock();
                        }
                    } catch (Exception e) {
                        Logger.INSTANCE.logln(Level.FATAL, Origin.INTAKE, String.format("Failed to write class %s", new Ansi().c(RED).s(value.name)));
                    }
                });

                resources.forEach(resource -> {
                    try {
                        JarEntry entry = new JarEntry(resource.name());
                        jos.putNextEntry(entry);
                        jos.write(resource.content());
                        jos.closeEntry();
                    } catch (Exception e) {
                        Logger.INSTANCE.logln(Level.FATAL, Origin.INTAKE, String.format("Failed to write resource %s", new Ansi().c(RED).s(resource.name())));
                    }
                });

            } catch (Exception e) {
                throw new RuntimeException(e);
            }

            try (FileOutputStream fos = new FileOutputStream(out.toFile())) {
                baos.writeTo(fos);

                File input = in.toFile();
                long inputLength = input.length();

                long outputLength = out.toFile().length();
                double ratio = ((double) outputLength - (double) inputLength) / (double) inputLength;

                Logger.INSTANCE.logln(Level.INFO, Origin.INTAKE,
                        String.format("File size changed by %s, (%s -> %s, %s)", new Ansi().c(WHITE).s(String.format("%.2f%%", ratio * 100)),
                                new Ansi().c(WHITE).s(String.format("%.2fKB", inputLength / 1024D)),
                                new Ansi().c(WHITE).s(String.format("%.2fKB", outputLength / 1024D)),
                                new Ansi().c(WHITE).s(String.format("%s%.2fKB",  ratio > 0 ? "+" : "-", Math.abs(outputLength - inputLength) / 1024D))));

            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
