package war.jar;

import com.google.gson.*;
import lombok.Getter;
import lombok.SneakyThrows;
import org.apache.commons.io.IOUtils;
import war.configuration.ConfigurationSection;
import war.jnt.dash.Ansi;
import war.jnt.dash.Level;
import war.jnt.dash.Logger;
import war.jnt.dash.Origin;
import war.metaphor.tree.JClassNode;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import static java.nio.file.Files.walk;
import static war.jnt.dash.Ansi.Color.*;

@Getter
public class JarReader {

    private Set<JClassNode> classes;
    private Set<JClassNode> libraries;
    private Set<JarResource> resources;

    private File input;
    private List<File> libs;

    private Manifest manifest;

    @SneakyThrows
    public void load(String path, ConfigurationSection config) {

        Logger logger = Logger.INSTANCE;

        this.classes = new HashSet<>();
        this.libraries = new HashSet<>();
        this.resources = new HashSet<>();

        this.input = new File(path);
        this.libs = new ArrayList<>();

        for (String lib : config.getStringList("libraries")) {
            File file = new File(lib);
            this.libs.add(file);
        }

        long start = System.currentTimeMillis();
        loadInput();

        logger.logln(Level.INFO, Origin.INTAKE, String.format("Loaded input (%s)", new Ansi().c(WHITE).s(String.format("%dms", System.currentTimeMillis() - start))));

        start = System.currentTimeMillis();
        loadLibraries();

        logger.logln(Level.INFO, Origin.INTAKE, String.format("Loaded libraries (%s)", new Ansi().c(WHITE).s(String.format("%dms", System.currentTimeMillis() - start))));

        List<String> mappingFiles = config.getStringList("mappings");

        if (mappingFiles != null && !mappingFiles.isEmpty()) {

            logger.logln(Level.INFO, Origin.METAPHOR, String.format("Loading mappings from %s", new Ansi().c(WHITE).s(mappingFiles)));

            Gson gson = new GsonBuilder().create();

            for (String mappingPath : mappingFiles) {
                File mappingFile = new File(mappingPath);
                if (!mappingFile.exists()) {
                    logger.logln(Level.WARNING, Origin.METAPHOR, String.format("Mapping file %s does not exist!", new Ansi().c(YELLOW).s(mappingPath).r(false).c(BRIGHT_YELLOW)));
                    continue;
                }

                byte[] data = IOUtils.toByteArray(mappingFile.toURI());
                String content = new String(data);

                JsonObject mappings = gson.fromJson(content, JsonObject.class);
                JsonObject classes = mappings.getAsJsonObject("classes");

                if (classes == null) {
                    logger.logln(Level.WARNING, Origin.METAPHOR, String.format("Mapping file %s does not contain classes!", new Ansi().c(YELLOW).s(mappingPath).r(false).c(BRIGHT_YELLOW)));
                    continue;
                }

                for (Map.Entry<String, JsonElement> entry : classes.entrySet()) {
                    String original = entry.getKey();
                    JsonPrimitive mapped = entry.getValue().getAsJsonPrimitive();
                    String mappedName = mapped.getAsString();
                    JClassNode classNode = findClass(mappedName);
                    if (classNode == null) {
                        logger.logln(Level.WARNING, Origin.METAPHOR, String.format("Class %s not found!", new Ansi().c(YELLOW).s(original).r(false).c(BRIGHT_YELLOW)));
                        continue;
                    }
                    classNode.setRealName(original);
                }

            }

        }

        for (JClassNode classNode : classes) {
            String sourceFile = classNode.sourceFile;
            if (sourceFile == null) continue;
            if (!sourceFile.startsWith("pass::jnt:")) continue;
            classNode.setRealName(sourceFile.substring(10));
        }
    }

    private JClassNode findClass(String name) {
        for (JClassNode classNode : classes) {
            if (classNode.name.equals(name)) {
                return classNode;
            }
        }
        for (JClassNode library : libraries) {
            if (library.name.equals(name)) {
                return library;
            }
        }
        return null;
    }

    private void loadInput() {
        if (!input.exists()) throw new RuntimeException("input file does not exist");
        if (!input.isFile()) throw new RuntimeException("input is not a file");
        load(input, false);
    }

    private void loadLibraries() {
        libs.parallelStream().forEach(lib -> {
            if (!lib.isDirectory()) if (lib.getName().endsWith(".jar")) load(lib, true);
            try {
                try (var stream = walk(lib.toPath())) {
                    stream.filter(path -> path.toString().endsWith(".jar"))
                            .forEach(path -> load(path.toFile(), true));
                }
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    @SneakyThrows
    private void load(File file, boolean library) {
        try (var reader = new JarIterator(file, library)) {
            while (reader.hasNext()) {
                JarContent<?> content = reader.next();
                if (content == null) continue;
                if (content.content() instanceof JClassNode cw) {
                    if (cw.isLibrary()) {
                        libraries.add(cw);
                    } else {
                        classes.add(cw);
                    }
                } else if (content.content() instanceof JarResource resource) {
                    if (!library) {
                        if (resource.name().equals(JarFile.MANIFEST_NAME)) {
                            manifest = new Manifest(new ByteArrayInputStream(resource.content()));
                        } else {
                            resources.add(resource);
                        }
                    }
                }

            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void clear() {
        if (classes != null) {
            classes.clear();
        }
        if (libraries != null) {
            libraries.clear();
        }
        if (resources != null) {
            resources.clear();
        }
        if (libs != null) {
            libs.clear();
        }
        input = null;
        manifest = null;
    }
}
