package war.jar;

import lombok.SneakyThrows;
import org.apache.commons.io.IOUtils;
import org.objectweb.asm.ClassReader;
import war.metaphor.tree.JClassNode;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.Iterator;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

public class JarIterator implements Iterator<JarContent<?>>, Closeable {

    private final boolean library;
    private final ZipFile jar;

    private final ZipInputStream jis;

    @SneakyThrows
    public JarIterator(File jar, boolean library) {
        this.library = library;
        this.jar = new ZipFile(jar);
        this.jis = new ZipInputStream(Files.newInputStream(jar.toPath()));
    }

    public JarIterator(File jar) {
        this(jar, false);
    }

    @Override
    @SneakyThrows
    public boolean hasNext() {
        return jis.available() > 0;
    }

    @Override
    @SneakyThrows
    public JarContent<?> next() {

        ZipEntry entry = jis.getNextEntry();
        if (entry == null) return null;

        if (entry.isDirectory()) return next();

        byte[] bytes = null;
        try {
            bytes = IOUtils.toByteArray(jar.getInputStream(entry));

            if (entry.getName().endsWith(".class")) {
                ClassReader cr = new ClassReader(bytes);
                JClassNode cw = new JClassNode(library);
                try {
                    cr.accept(cw, ClassReader.SKIP_FRAMES);
                    return new JarContent<>(cw);
                } catch (Exception ex) {
                    return new JarContent<>(new JarResource(entry.getName(), bytes));
                } finally {
                    // Clear ClassReader reference to help GC
                    cr = null;
                }
            }

            return new JarContent<>(new JarResource(entry.getName(), bytes));
        } finally {
            // Explicitly null the bytes array to help GC
            bytes = null;
        }
    }

    @Override
    public void close() throws IOException {
        jis.close();
        jar.close();
    }
}
