package war.ice;

import java.util.List;

public abstract class Statement {
    public interface Visitor<R> {
        R visitBlockStmt(Block stmt);
        R visitExpressionStmt(Expression stmt);
        R visitFunctionStmt(Function stmt);
        R visitPrintStmt(Print stmt);
        R visitPrintlnStmt(Println stmt);
        R visitReturnStmt(Return stmt);
        R visitVarStmt(Var stmt);
        R visitIfStmt(If stmt);
        R visitImportStmt(Import stmt);
        R visitWhileStmt(While stmt);
    }

    public abstract <R> R accept(Visitor<R> visitor);

    public static class Block extends Statement {
        final List<Statement> statements;

        public Block(List<Statement> statements) {
            this.statements = statements;
        }

        @Override
        public <R> R accept(Visitor<R> visitor) {
            return visitor.visitBlockStmt(this);
        }
    }

    public static class Expression extends Statement {
        final war.ice.Expression expression;

        public Expression(war.ice.Expression expression) {
            this.expression = expression;
        }

        @Override
        public <R> R accept(Visitor<R> visitor) {
            return visitor.visitExpressionStmt(this);
        }
    }

    public static class Function extends Statement {
        final Token name;
        final List<Token> params;
        final List<Statement> body;

        public Function(Token name, List<Token> params, List<Statement> body) {
            this.name = name;
            this.params = params;
            this.body = body;
        }

        @Override
        public <R> R accept(Visitor<R> visitor) {
            return visitor.visitFunctionStmt(this);
        }
    }

    public static class Print extends Statement {
        public final war.ice.Expression expression;

        public Print(war.ice.Expression expression) {
            this.expression = expression;
        }

        @Override
        public <R> R accept(Visitor<R> visitor) {
            return visitor.visitPrintStmt(this);
        }
    }

    public static class Println extends Statement {
        public final war.ice.Expression expression;

        public Println(war.ice.Expression expression) {
            this.expression = expression;
        }

        @Override
        public <R> R accept(Visitor<R> visitor) {
            return visitor.visitPrintlnStmt(this);
        }
    }

    public static class Return extends Statement {
        final Token keyword;
        final war.ice.Expression value;

        public Return(Token keyword, war.ice.Expression value) {
            this.keyword = keyword;
            this.value = value;
        }

        @Override
        public <R> R accept(Visitor<R> visitor) {
            return visitor.visitReturnStmt(this);
        }
    }

    public static class Var extends Statement {
        public final Token name;
        public final war.ice.Expression initializer;

        public Var(Token name, war.ice.Expression initializer) {
            this.name = name;
            this.initializer = initializer;
        }

        @Override
        public <R> R accept(Visitor<R> visitor) {
            return visitor.visitVarStmt(this);
        }
    }

    public static class If extends Statement {
        public final war.ice.Expression condition;
        public final Statement thenBranch;
        public final Statement elseBranch;

        public If(war.ice.Expression condition, Statement thenBranch, Statement elseBranch) {
            this.condition = condition;
            this.thenBranch = thenBranch;
            this.elseBranch = elseBranch;
        }

        @Override
        public <R> R accept(Visitor<R> visitor) {
            return visitor.visitIfStmt(this);
        }
    }

    public static class Import extends Statement {
        public final Token moduleName;

        public Import(Token moduleName) {
            this.moduleName = moduleName;
        }

        @Override
        public <R> R accept(Visitor<R> visitor) {
            return visitor.visitImportStmt(this);
        }
    }

    public static class While extends Statement {
        public final war.ice.Expression condition;
        public final Statement body;

        public While(war.ice.Expression condition, Statement body) {
            this.condition = condition;
            this.body = body;
        }

        @Override
        public <R> R accept(Visitor<R> visitor) {
            return visitor.visitWhileStmt(this);
        }
    }
} 