package war.ice;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class Lexer {
    private final String source;
    private int current = 0;
    private int start = 0;
    private static final Map<String, TokenType> keywords;

    static {
        keywords = new HashMap<>();
        keywords.put("var", TokenType.VAR);
        keywords.put("print", TokenType.PRINT);
        keywords.put("println", TokenType.PRINTLN);
        keywords.put("ret", TokenType.RET);
        keywords.put("inline", TokenType.INLINE);
        keywords.put("if", TokenType.IF);
        keywords.put("else", TokenType.ELSE);
        keywords.put("and", TokenType.AND);
        keywords.put("or", TokenType.OR);
        keywords.put("import", TokenType.IMPORT);
        keywords.put("while", TokenType.WHILE);
        keywords.put("for", TokenType.FOR);
    }

    public Lexer(String source) {
        this.source = source;
    }

    public List<Token> tokenize() {
        List<Token> tokens = new ArrayList<>();
        
        while (!isAtEnd()) {
            start = current;
            scanToken(tokens);
        }

        tokens.add(new Token(TokenType.EOF, "", null, 0));
        return tokens;
    }

    private void scanToken(List<Token> tokens) {
        char c = advance();
        switch (c) {
            case '(': addToken(tokens, TokenType.LEFT_PAREN); break;
            case ')': addToken(tokens, TokenType.RIGHT_PAREN); break;
            case '{': addToken(tokens, TokenType.LEFT_BRACE); break;
            case '}': addToken(tokens, TokenType.RIGHT_BRACE); break;
            case '[': addToken(tokens, TokenType.LEFT_BRACKET); break;
            case ']': addToken(tokens, TokenType.RIGHT_BRACKET); break;
            case ',': addToken(tokens, TokenType.COMMA); break;
            case '.': addToken(tokens, TokenType.DOT); break;
            case '-': 
                if (match('>')) {
                    addToken(tokens, TokenType.ARROW);
                } else {
                    addToken(tokens, TokenType.MINUS);
                }
                break;
            case '+': addToken(tokens, TokenType.PLUS); break;
            case ';': addToken(tokens, TokenType.SEMICOLON); break;
            case '*': addToken(tokens, TokenType.STAR); break;
            case '%': addToken(tokens, TokenType.PERCENT); break;
            case '!': 
                addToken(tokens, match('=') ? TokenType.BANG_EQUAL : TokenType.BANG);
                break;
            case '=': 
                addToken(tokens, match('=') ? TokenType.EQUAL_EQUAL : TokenType.EQUAL);
                break;
            case '<': 
                addToken(tokens, match('=') ? TokenType.LESS_EQUAL : TokenType.LESS);
                break;
            case '>': 
                addToken(tokens, match('=') ? TokenType.GREATER_EQUAL : TokenType.GREATER);
                break;
            case '/': 
                if (match('/')) {
                    while (peek() != '\n' && !isAtEnd()) advance();
                } else {
                    addToken(tokens, TokenType.SLASH);
                }
                break;
            case '@': 
                if (match('i')) {
                    if (match('m') && match('p') && match('o') && match('r') && match('t')) {
                        addToken(tokens, TokenType.IMPORT);
                    } else if (match('n') && match('l') && match('i') && match('n') && match('e')) {
                        addToken(tokens, TokenType.INLINE);
                    } else {
                        throw new RuntimeException("Unexpected character after @");
                    }
                } else {
                    throw new RuntimeException("Unexpected character after @");
                }
                break;
            case ':':
                if (match(':')) {
                    addToken(tokens, TokenType.COLON_COLON);
                }
                break;
            case '"': string(tokens); break;
            case '$': addToken(tokens, TokenType.DOLLAR); break;
            default:
                if (isDigit(c)) {
                    number(tokens);
                } else if (isAlpha(c)) {
                    identifier(tokens);
                } else if (!isWhitespace(c)) {
                    throw new RuntimeException("Unexpected character: " + c);
                }
                break;
        }
    }

    private boolean isAtEnd() {
        return current >= source.length();
    }

    private char advance() {
        return source.charAt(current++);
    }

    private void addToken(List<Token> tokens, TokenType type) {
        addToken(tokens, type, null);
    }

    private void addToken(List<Token> tokens, TokenType type, Object literal) {
        String text = source.substring(start, current);
        tokens.add(new Token(type, text, literal, 0));
    }

    private boolean isDigit(char c) {
        return c >= '0' && c <= '9';
    }

    private boolean isAlpha(char c) {
        return (c >= 'a' && c <= 'z') ||
               (c >= 'A' && c <= 'Z') ||
               c == '_';
    }

    private boolean isWhitespace(char c) {
        return c == ' ' || c == '\r' || c == '\t' || c == '\n';
    }

    private void number(List<Token> tokens) {
        while (isDigit(peek())) advance();

        if (peek() == '.' && isDigit(peekNext())) {
            advance();
            while (isDigit(peek())) advance();
        }

        addToken(tokens, TokenType.NUMBER,
            Double.parseDouble(source.substring(start, current)));
    }

    private void identifier(List<Token> tokens) {
        while (isAlphaNumeric(peek())) advance();
        
        String text = source.substring(start, current);
        TokenType type = keywords.getOrDefault(text, TokenType.IDENTIFIER);
        addToken(tokens, type);
    }

    private boolean isAlphaNumeric(char c) {
        return isAlpha(c) || isDigit(c);
    }

    private char peek() {
        if (isAtEnd()) return '\0';
        return source.charAt(current);
    }

    private char peekNext() {
        if (current + 1 >= source.length()) return '\0';
        return source.charAt(current + 1);
    }

    private boolean match(char expected) {
        if (isAtEnd()) return false;
        if (source.charAt(current) != expected) return false;
        current++;
        return true;
    }

    private void string(List<Token> tokens) {
        StringBuilder value = new StringBuilder();
        while (peek() != '"' && !isAtEnd()) {
            if (peek() == '$') {
                advance();
                if (isAlpha(peek())) {
                    int varStart = current;
                    while (isAlphaNumeric(peek())) advance();
                    String varName = source.substring(varStart, current);
                    value.append("$").append(varName);
                } else {
                    value.append('$');
                }
            } else {
                value.append(advance());
            }
        }

        if (isAtEnd()) {
            throw new RuntimeException("Unterminated string.");
        }

        advance();
        addToken(tokens, TokenType.STRING, value.toString());
    }
} 