package org.jnt.matrix.util;

import org.jnt.matrix.base.DeobfBase;
import org.jnt.matrix.logger.LogLevel;
import org.jnt.matrix.logger.Logger;
import org.jnt.matrix.wrapper.ClassWrapper;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.tree.ClassNode;
import software.coley.cafedude.InvalidClassException;
import software.coley.cafedude.io.ClassFileReader;
import software.coley.cafedude.io.ClassFileWriter;
import software.coley.lljzip.ZipIO;
import software.coley.lljzip.format.compression.ZipCompressions;
import software.coley.lljzip.format.model.*;

import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.foreign.ValueLayout;
import java.nio.file.Path;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * Loads and saves files
 * @author marie
 */
public final class FileUtil {

    /**
     * Loads the zip file using LL-Java-Zip
     *
     * @param path The path to the zip file to load
     */
    public static void loadJar(final Path path) {
        try (ZipArchive archive = ZipIO.readJvm(path)) {
            List<LocalFileHeader> localFiles = archive.getLocalFiles();

            for (LocalFileHeader localFile : localFiles) {
                byte[] bytes = ZipCompressions.decompress(localFile).toArray(ValueLayout.OfByte.JAVA_BYTE);

                try {
                    byte[] patchedBytes = new ClassFileWriter().write(new ClassFileReader().read(bytes));

                    ClassReader classReader = new ClassReader(bytes);
                    ClassNode c = new ClassNode();
                    classReader.accept(c, DeobfBase.INSTANCE.readerArgs);
                    DeobfBase.INSTANCE.classes.put(c.name, new ClassWrapper(patchedBytes, c));
                } catch (InvalidClassException invalidClassException) {
                    // assume that the class we loaded just isn't a class

                    DeobfBase.INSTANCE.files.put(localFile.getFileNameAsString(), bytes);
                }
            }
        } catch (IOException ioException) {
            Logger.log(LogLevel.FAIL, "Failed to load the zip file! Reason: ", ioException.getMessage());
        }
    }

    /**
     * Saves the jar
     *
     * @param path The path to save the jar to
     */
    public static void saveJar(final Path path) {
        try (
            FileOutputStream fos = new FileOutputStream(path.toFile());
            ZipOutputStream zipOut = new ZipOutputStream(fos);
        ) {
            for (ClassWrapper classWrapper : DeobfBase.INSTANCE.classes.values()) {
                ZipEntry zipEntry = new ZipEntry(classWrapper.base.name + ".class");
                zipOut.putNextEntry(zipEntry);

                ClassWriter classWriter = new ClassWriter(DeobfBase.INSTANCE.writerArgs);
                classWrapper.prepareForWrite();
                classWrapper.base.accept(classWriter);

                zipOut.write(classWriter.toByteArray());
            }
        } catch (Throwable t) {
            throw new RuntimeException(t);
        }
    }

}
