package org.jnt.matrix.ir.stmt.impl;

import org.jnt.matrix.ir.asm.InsnBuilder;
import org.jnt.matrix.ir.codegen.CompilerSheet;
import org.jnt.matrix.ir.expr.IExpr;
import org.jnt.matrix.ir.print.IRPrinter;
import org.jnt.matrix.ir.stmt.IStmt;
import org.objectweb.asm.tree.LabelNode;

import static org.objectweb.asm.Opcodes.*;

/**
 * @author etho
 */
public class IfStmt implements IStmt {
    private final IExpr[] exprs;
    private final int opcode;
    private final LabelNode target;

    public IfStmt(int opcode, LabelNode target, IExpr... exprs) {
        this.exprs = exprs;
        this.opcode = opcode;
        this.target = target;
    }

    @Override
    public void compile(CompilerSheet sheet) {
        InsnBuilder b = new InsnBuilder();

        switch (opcode) {
            case IFEQ, IFNE -> {
                exprs[0].compile(sheet);

                if (opcode == IFEQ) b.ifeq(target);
                else b.ifne(target);
            }
            case IF_ICMPEQ, IF_ICMPNE, IF_ICMPGT, IF_ICMPLT, IF_ICMPGE, IF_ICMPLE -> {
                exprs[1].compile(sheet);
                exprs[0].compile(sheet);

                switch (opcode) {
                    case IF_ICMPEQ -> b.if_icmpeq(target);
                    case IF_ICMPNE -> b.if_icmpne(target);
                    case IF_ICMPGT -> b.if_icmpgt(target);
                    case IF_ICMPLT -> b.if_icmplt(target);
                    case IF_ICMPGE -> b.if_icmpge(target);
                    case IF_ICMPLE -> b.if_icmple(target);
                }
            }
        }

        sheet.register(b.build());
    }

    @Override
    public void print(IRPrinter printer) {
        printer.tab().out("if (");

        switch (opcode) {
            case IFEQ, IFNE -> {
                printer.out(opcode == IFEQ ? "!" : "");
                exprs[0].print(printer);
            }
            case IF_ICMPEQ, IF_ICMPNE -> {
                IExpr a = exprs[0];
                IExpr b = exprs[1];

                b.print(printer);

                printer.space();
                printer.out(opcode == IF_ICMPEQ ? "==" : "!=").space();

                a.print(printer);
            }
        }

        printer.out(")").nl().tab().tab().out("jmp ");
        printer.out("b" + target.getLabel().getOffset()).nl();
    }
}
