package org.jnt.matrix.ir.asm;

import org.objectweb.asm.Opcodes;
import org.objectweb.asm.tree.*;

import static org.objectweb.asm.Opcodes.*;

/**
 * @author etho
 */
public class InsnBuilder {
    private final InsnList instructions = new InsnList();

    public InsnList build() {
        return instructions;
    }

    public InsnBuilder nullCst() {
        instructions.add(new InsnNode(ACONST_NULL));
        return this;
    }

    public InsnBuilder pushInt(int value) {
        instructions.add(switch (value) {
            case -1 -> new InsnNode(ICONST_M1);
            case 0 -> new InsnNode(ICONST_0);
            case 1 -> new InsnNode(ICONST_1);
            case 2 -> new InsnNode(ICONST_2);
            case 3 -> new InsnNode(ICONST_3);
            case 4 -> new InsnNode(ICONST_4);
            case 5 -> new InsnNode(ICONST_5);
            default -> {
                if (value >= Byte.MIN_VALUE && value <= Byte.MAX_VALUE) {
                    yield new IntInsnNode(BIPUSH, value);
                } else if (value >= Short.MIN_VALUE && value <= Short.MAX_VALUE) {
                    yield new IntInsnNode(SIPUSH, value);
                } else {
                    yield new LdcInsnNode(value);
                }
            }
        });
        return this;
    }

    public InsnBuilder ldc(Object cst) {
        instructions.add(new LdcInsnNode(cst));
        return this;
    }

    public InsnBuilder dup() {
        instructions.add(new InsnNode(DUP));
        return this;
    }

    public InsnBuilder ireturn() {
        instructions.add(new InsnNode(IRETURN));
        return this;
    }

    public InsnBuilder freturn() {
        instructions.add(new InsnNode(FRETURN));
        return this;
    }

    public InsnBuilder dreturn() {
        instructions.add(new InsnNode(DRETURN));
        return this;
    }

    public InsnBuilder lreturn() {
        instructions.add(new InsnNode(LRETURN));
        return this;
    }

    public InsnBuilder areturn() {
        instructions.add(new InsnNode(ARETURN));
        return this;
    }

    public InsnBuilder istore(int index) {
        instructions.add(new VarInsnNode(ISTORE, index));
        return this;
    }

    public InsnBuilder fstore(int index) {
        instructions.add(new VarInsnNode(FSTORE, index));
        return this;
    }

    public InsnBuilder dstore(int index) {
        instructions.add(new VarInsnNode(DSTORE, index));
        return this;
    }

    public InsnBuilder lstore(int index) {
        instructions.add(new VarInsnNode(LSTORE, index));
        return this;
    }

    public InsnBuilder astore(int index) {
        instructions.add(new VarInsnNode(ASTORE, index));
        return this;
    }

    public InsnBuilder iload(int index) {
        instructions.add(new VarInsnNode(ILOAD, index));
        return this;
    }

    public InsnBuilder fload(int index) {
        instructions.add(new VarInsnNode(FLOAD, index));
        return this;
    }

    public InsnBuilder dload(int index) {
        instructions.add(new VarInsnNode(DLOAD, index));
        return this;
    }

    public InsnBuilder lload(int index) {
        instructions.add(new VarInsnNode(LLOAD, index));
        return this;
    }

    public InsnBuilder aload(int index) {
        instructions.add(new VarInsnNode(ALOAD, index));
        return this;
    }

    public InsnBuilder pushLong(long value) {
        instructions.add(switch ((int) value) {
            case 0 -> new InsnNode(LCONST_0);
            case 1 -> new InsnNode(LCONST_1);
            default -> new LdcInsnNode(value);
        });
        return this;
    }

    public InsnBuilder iadd() {
        instructions.add(new InsnNode(IADD));
        return this;
    }

    public InsnBuilder fadd() {
        instructions.add(new InsnNode(FADD));
        return this;
    }

    public InsnBuilder dadd() {
        instructions.add(new InsnNode(DADD));
        return this;
    }

    public InsnBuilder ladd() {
        instructions.add(new InsnNode(LADD));
        return this;
    }

    public InsnBuilder isub() {
        instructions.add(new InsnNode(ISUB));
        return this;
    }

    public InsnBuilder fsub() {
        instructions.add(new InsnNode(FSUB));
        return this;
    }

    public InsnBuilder dsub() {
        instructions.add(new InsnNode(DSUB));
        return this;
    }

    public InsnBuilder lsub() {
        instructions.add(new InsnNode(LSUB));
        return this;
    }

    public InsnBuilder imul() {
        instructions.add(new InsnNode(IMUL));
        return this;
    }

    public InsnBuilder fmul() {
        instructions.add(new InsnNode(FMUL));
        return this;
    }

    public InsnBuilder dmul() {
        instructions.add(new InsnNode(DMUL));
        return this;
    }

    public InsnBuilder lmul() {
        instructions.add(new InsnNode(LMUL));
        return this;
    }

    public InsnBuilder idiv() {
        instructions.add(new InsnNode(IDIV));
        return this;
    }

    public InsnBuilder fdiv() {
        instructions.add(new InsnNode(FDIV));
        return this;
    }

    public InsnBuilder ddiv() {
        instructions.add(new InsnNode(DDIV));
        return this;
    }

    public InsnBuilder ldiv() {
        instructions.add(new InsnNode(LDIV));
        return this;
    }

    public InsnBuilder irem() {
        instructions.add(new InsnNode(IREM));
        return this;
    }

    public InsnBuilder frem() {
        instructions.add(new InsnNode(FREM));
        return this;
    }

    public InsnBuilder drem() {
        instructions.add(new InsnNode(DREM));
        return this;
    }

    public InsnBuilder lrem() {
        instructions.add(new InsnNode(LREM));
        return this;
    }

    public InsnBuilder ineg() {
        instructions.add(new InsnNode(INEG));
        return this;
    }

    public InsnBuilder fneg() {
        instructions.add(new InsnNode(FNEG));
        return this;
    }

    public InsnBuilder dneg() {
        instructions.add(new InsnNode(DNEG));
        return this;
    }

    public InsnBuilder lneg() {
        instructions.add(new InsnNode(LNEG));
        return this;
    }

    public InsnBuilder ixor() {
        instructions.add(new InsnNode(IXOR));
        return this;
    }

    public InsnBuilder lxor() {
        instructions.add(new InsnNode(LXOR));
        return this;
    }

    public InsnBuilder ishl() {
        instructions.add(new InsnNode(ISHL));
        return this;
    }

    public InsnBuilder lshl() {
        instructions.add(new InsnNode(LSHL));
        return this;
    }

    public InsnBuilder ishr() {
        instructions.add(new InsnNode(ISHR));
        return this;
    }

    public InsnBuilder lshr() {
        instructions.add(new InsnNode(LSHR));
        return this;
    }

    public InsnBuilder iushr() {
        instructions.add(new InsnNode(IUSHR));
        return this;
    }

    public InsnBuilder lushr() {
        instructions.add(new InsnNode(LUSHR));
        return this;
    }

    public InsnBuilder ior() {
        instructions.add(new InsnNode(IOR));
        return this;
    }

    public InsnBuilder lor() {
        instructions.add(new InsnNode(LOR));
        return this;
    }

    public InsnBuilder iand() {
        instructions.add(new InsnNode(IAND));
        return this;
    }

    public InsnBuilder land() {
        instructions.add(new InsnNode(LAND));
        return this;
    }

    public InsnBuilder _return() {
        instructions.add(new InsnNode(RETURN));
        return this;
    }

    public InsnBuilder ifeq(LabelNode target) {
        instructions.add(new JumpInsnNode(IFEQ, target));
        return this;
    }

    public InsnBuilder ifne(LabelNode target) {
        instructions.add(new JumpInsnNode(IFNE, target));
        return this;
    }

    public InsnBuilder if_icmpeq(LabelNode target) {
        instructions.add(new JumpInsnNode(IF_ICMPEQ, target));
        return this;
    }

    public InsnBuilder if_icmpne(LabelNode target) {
        instructions.add(new JumpInsnNode(IF_ICMPNE, target));
        return this;
    }

    public InsnBuilder if_icmpgt(LabelNode target) {
        instructions.add(new JumpInsnNode(IF_ICMPGT, target));
        return this;
    }

    public InsnBuilder if_icmplt(LabelNode target) {
        instructions.add(new JumpInsnNode(IF_ICMPLT, target));
        return this;
    }

    public InsnBuilder if_icmpge(LabelNode target) {
        instructions.add(new JumpInsnNode(IF_ICMPGE, target));
        return this;
    }

    public InsnBuilder if_icmple(LabelNode target) {
        instructions.add(new JumpInsnNode(IF_ICMPLE, target));
        return this;
    }

    public InsnBuilder pushFloat(float value) {
        instructions.add(switch ((int) value) {
            case 0 -> new InsnNode(FCONST_0);
            case 1 -> new InsnNode(FCONST_1);
            default -> new LdcInsnNode(value);
        });
        return this;
    }

    public InsnBuilder pushDouble(double value) {
        instructions.add(switch ((int) value) {
            case 0 -> new InsnNode(DCONST_0);
            case 1 -> new InsnNode(DCONST_1);
            default -> new LdcInsnNode(value);
        });
        return this;
    }
}
