#version 450 core

layout(std140, binding = 0) uniform ShaderUBO {
    vec2 resolution;
    vec2 loc;
    vec2 size;
};

uniform float radius;
uniform float angle;
uniform int interpolationType;
uniform float outlineWidth;
uniform vec4 colors[5];
uniform int colorCount;

out vec4 FragColor;

float colorStops[5];

float ease(float t) {
    return t < 0.5 ? 2.0 * t * t : 1.0 - pow(-2.0 * t + 2.0, 2.0) / 2.0;
}

float roundBox(vec2 pos, vec2 halfSize, float r) {
    vec2 d = abs(pos) - halfSize + vec2(r);
    return min(max(d.x, d.y), 0.0) + length(max(d, 0.0)) - r;
}

vec4 mixColors(vec4 c1, vec4 c2, float t) {
    return mix(c1, c2, t);
}

vec4 getGradientColor(float t) {
    t = clamp(t, 0.0, 1.0);
    if (colorCount < 2) return colors[0];
    if (t <= colorStops[0]) return colors[0];
    for (int i = 0; i < colorCount - 1; i++) {
        if (t >= colorStops[i] && t <= colorStops[i + 1]) {
            float localT = (t - colorStops[i]) / (colorStops[i + 1] - colorStops[i]);
            if (interpolationType == 1) localT = ease(localT);
            return mixColors(colors[i], colors[i + 1], localT);
        }
    }
    return colors[colorCount - 1];
}

void main() {
    for (int i = 0; i < colorCount; i++) {
        colorStops[i] = float(i) / float(colorCount - 1);
    }

    vec2 halfSize = size * 0.5;
    vec2 pos = gl_FragCoord.xy - loc - halfSize;
    float mask = roundBox(pos, halfSize, radius);
    float distance = fwidth(mask) * 0.5;
    float alpha = smoothstep(distance, -distance, mask);

    if (alpha <= 0.0) {
        discard;
    }

    vec2 normalizedPos = (gl_FragCoord.xy - loc) / size;
    float rad = radians(angle);
    vec2 dir = vec2(cos(rad), sin(rad));
    float t = (normalizedPos.x * dir.x + normalizedPos.y * dir.y + 1.0) * 0.5;
    t = clamp(t, 0.0, 1.0);

    vec4 gradientColor = getGradientColor(t);

    if (outlineWidth > 0.0) {
        float innerMask = roundBox(pos, halfSize - vec2(outlineWidth), radius - outlineWidth);
        float innerAlpha = smoothstep(distance, -distance, innerMask);
        if (innerAlpha > 0.0) {
            FragColor = vec4(gradientColor.rgb, gradientColor.a * alpha);
        } else {
            FragColor = vec4(gradientColor.rgb, gradientColor.a * alpha);
        }
    } else {
        FragColor = vec4(gradientColor.rgb, gradientColor.a * alpha);
    }
}