#version 450 core

layout(std140, binding = 0) uniform ShaderUBO {
    vec2 resolution;
    vec2 loc;
    vec2 size;
};

uniform float angle;
uniform int interpolationType;
uniform float outlineWidth;
uniform vec4 colors[5];
uniform int colorCount;

out vec4 FragColor;

float colorStops[5];

float ease(float t) {
    return t < 0.5 ? 2.0 * t * t : 1.0 - pow(-2.0 * t + 2.0, 2.0) / 2.0;
}

float dither(vec2 uv, float alpha) {
    float d = fract(sin(dot(uv, vec2(12.9898, 78.233))) * 43758.5453);
    return step(d, alpha);
}

float noise(vec2 p) {
    return fract(sin(dot(p, vec2(127.1, 311.7))) * 43758.5453);
}

vec4 mixColors(vec4 c1, vec4 c2, float t) {
    return mix(c1, c2, t);
}

vec4 getGradientColor(float t) {
    t = clamp(t, 0.0, 1.0);
    if (colorCount < 2) return colors[0];
    if (t <= colorStops[0]) return colors[0];
    for (int i = 0; i < colorCount - 1; i++) {
        if (t >= colorStops[i] && t <= colorStops[i + 1]) {
            float localT = (t - colorStops[i]) / (colorStops[i + 1] - colorStops[i]);
            if (interpolationType == 1) localT = ease(localT);
            return mixColors(colors[i], colors[i + 1], localT);
        }
    }
    return colors[colorCount - 1];
}

void main() {
    for (int i = 0; i < colorCount; i++) {
        colorStops[i] = float(i) / float(colorCount - 1);
    }

    vec2 pos = gl_FragCoord.xy;
    vec2 rectMin = loc;
    vec2 rectMax = loc + size;
    float insideRectX = step(rectMin.x, pos.x) * step(pos.x, rectMax.x);
    float insideRectY = step(rectMin.y, pos.y) * step(pos.y, rectMax.y);
    float insideRect = insideRectX * insideRectY;

    if (insideRect > 0.0) {
        vec2 normalizedPos = (pos - rectMin) / size;
        float rad = radians(angle);
        vec2 dir = vec2(cos(rad), sin(rad));
        float t = (normalizedPos.x * dir.x + normalizedPos.y * dir.y + 1.0) * 0.5;
        t = clamp(t, 0.0, 1.0);

        vec4 gradientColor = getGradientColor(t);
        float n = noise(pos * 0.01) * 0.02;
        gradientColor.rgb += vec3(n, n, n);

        float alpha = gradientColor.a;
        gradientColor.a = 1.0;
        if (alpha < 1.0) {
            float dithered = dither(pos / resolution, alpha);
            if (dithered < 0.5) discard;
        }

        if (outlineWidth > 0.0) {

    float distLeft = abs(pos.x - rectMin.x);
    float distRight = abs(pos.x - rectMax.x);
    float distTop = abs(pos.y - rectMin.y);
    float distBottom = abs(pos.y - rectMax.y);
    float minDist = min(min(distLeft, distRight), min(distTop, distBottom));

    if (minDist < outlineWidth) {
    FragColor = gradientColor;
    } else {
    discard;
    }
    } else {
    FragColor = gradientColor;
    }
    } else {
        discard;
    }

}