package tech.atani.client.util.game.render.shader.data.framebuffer;

import net.minecraft.client.renderer.GlStateManager;

import java.nio.IntBuffer;

import static org.lwjgl.opengl.GL45C.*;

public class FrameBufferImpl implements FrameBuffer {

    public final boolean depth;
    public final boolean hdr;
    private final float[] clearColor;

    private int width;
    private int height;

    private int colorAttachment = -1;
    private int depthAttachment = -1;

    private int fbo = -1;
    private int depthBuffer = -1;
    private FilterMode filter = null;

    public FrameBufferImpl(int width, int height, boolean depth, boolean hdr) {
        this.width = width;
        this.height = height;
        this.depth = depth;
        this.hdr = hdr;

        clearColor = new float[]{0.0f, 0.0f, 0.0f, 0.0f};

        create();
    }

    @Override
    public int handle() {
        return fbo;
    }

    @Override
    public void viewport() {
        GlStateManager.viewport(0, 0, width, height);
    }

    @Override
    public void beginWrite() {
        glBindFramebuffer(GL_FRAMEBUFFER, fbo);
    }

    @Override
    public void beginReadColorAttachment() {
        glBindTexture(GL_TEXTURE_2D, colorAttachment);
    }

    @Override
    public void beginReadDepthAttachment() {
        glBindTexture(GL_TEXTURE_2D, depthAttachment);
    }

    @Override
    public void endRead() {
        glBindTexture(GL_TEXTURE_2D, 0);
    }

    @Override
    public void endWrite() {
        glBindFramebuffer(GL_FRAMEBUFFER, 0);
    }

    @Override
    public void clear(boolean depth) {
        glClearColor(clearColor[0], clearColor[1], clearColor[2], clearColor[3]);

        int mask = GL_COLOR_BUFFER_BIT;
        if (depth) {
            mask |= GL_DEPTH_BUFFER_BIT;
        }

        glClearStencil(0);
        glClear(mask);
    }

    @Override
    public void setClearColor(float r, float g, float b, float a) {
        clearColor[0] = r;
        clearColor[1] = g;
        clearColor[2] = b;
        clearColor[3] = a;
    }

    @Override
    public void setFilter(FilterMode filter) {
        if (this.filter == filter) {
            return;
        }

        this.filter = filter;

        beginReadColorAttachment();

        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, filter.glType());
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, filter.glType());

        endRead();
    }

    @Override
    public void setWrap(TextureWrap wrap) {
        beginReadColorAttachment();

        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, wrap.glType());
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, wrap.glType());

        endRead();
    }

    @Override
    public void resize(int width, int height) {
        this.width = width;
        this.height = height;

        free();
        create();
    }

    @Override
    public void resizeIfNeeded(int width, int height) {
        width = Math.max(width, 1);
        height = Math.max(height, 1);

        if (this.width != width || this.height != height) {
            resize(width, height);
        }

    }

    @Override
    public void free() {
        endWrite();
        endRead();

        if (depthBuffer != -1) {
            glDeleteRenderbuffers(depthBuffer);
            depthBuffer = -1;
        }

        if (colorAttachment != -1) {
            glDeleteTextures(colorAttachment);
            colorAttachment = -1;
        }

        if (depthAttachment != -1) {
            glDeleteTextures(depthAttachment);
            depthAttachment = -1;
        }

        if (fbo != -1) {
            glBindFramebuffer(GL_FRAMEBUFFER, 0);
            glDeleteFramebuffers(fbo);
            fbo = -1;
        }
    }

    @Override
    public void copyDepth(FrameBuffer frameBuffer) {
        if (!(frameBuffer instanceof FrameBufferImpl fb)) {
            throw new IllegalArgumentException("FrameBuffer must be an instance of FrameBufferImpl");
        }

        glEnable(GL_DEPTH_TEST);
        glBindFramebuffer(GL_READ_FRAMEBUFFER, fb.fbo);
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER, fbo);
        glBlitFramebuffer(
                0, 0, fb.width, fb.height, 0, 0, fb.width, fb.height, GL_DEPTH_BUFFER_BIT, GL_NEAREST);
        glBindFramebuffer(GL_FRAMEBUFFER, 0);
    }

    @Override
    public int width() {
        return width;
    }

    @Override
    public int height() {
        return height;
    }

    @Override
    public int colorAttachment() {
        return colorAttachment;
    }

    @Override
    public int depthAttachment() {
        return depthAttachment;
    }

    private void create() {
        fbo = glGenFramebuffers();
        colorAttachment = glGenTextures();

        if (depth) {
            GlStateManager.enableDepth();

            depthBuffer = glGenRenderbuffers();
            depthAttachment = glGenTextures();

            beginReadDepthAttachment();
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE, 0);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
            glTexImage2D(
                    GL_TEXTURE_2D,
                    0,
                    GL_DEPTH32F_STENCIL8,
                    width,
                    height,
                    0,
                    GL_DEPTH_STENCIL,
                    GL_FLOAT_32_UNSIGNED_INT_24_8_REV,
                    (IntBuffer) null);
        }

        filter = null;

        setFilter(FilterMode.NEAREST);
        beginReadColorAttachment();

        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
        glTexImage2D(
                GL_TEXTURE_2D,
                0,
                hdr ? GL_RGBA16 : GL_RGBA8,
                width,
                height,
                0,
                GL_RGBA,
                GL_UNSIGNED_BYTE,
                (IntBuffer) null);

        beginWrite();
        glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, colorAttachment, 0);

        if (depth) {
            glFramebufferTexture2D(
                    GL_FRAMEBUFFER, GL_DEPTH_STENCIL_ATTACHMENT, GL_TEXTURE_2D, depthAttachment, 0);
        }

        clear(true);
        endRead();

        endWrite();
    }
}
