package tech.atani.client.util.game.render.font.gui.shader;

import org.lwjgl.BufferUtils;

import java.nio.FloatBuffer;

import static net.minecraft.client.renderer.OpenGlHelper.glUniformMatrix4;
import static org.lwjgl.opengl.GL11.*;
import static org.lwjgl.opengl.GL20.*;

public abstract class ShaderProgram {
    protected final int name;

    protected final FloatBuffer modelView;
    protected final FloatBuffer projection;

    protected final int modelViewUniform;
    protected final int projectionUniform;

    protected boolean updateMatrices;

    public ShaderProgram(FloatBuffer modelView, FloatBuffer projection, Shader... shaders) {
        this.modelView = modelView;
        this.projection = projection;

        name = glCreateProgram();
        for (Shader shader : shaders) {
            shader.attach(name);
        }

        bindAttribLocation(name);

        glLinkProgram(name);

        for (Shader shader : shaders) {
            shader.detach(name);
            shader.delete();
        }

        if (glGetProgrami(name, GL_LINK_STATUS) == 0) {
            int logLength = glGetProgrami(name, GL_INFO_LOG_LENGTH);
            throw new IllegalStateException("Failed to link program: " + glGetProgramInfoLog(name, logLength));
        }

        modelViewUniform = glGetUniformLocation(name, "u_modelView");
        projectionUniform = glGetUniformLocation(name, "u_projection");
    }

    public ShaderProgram(Shader... shaders) {
        this(BufferUtils.createFloatBuffer(16), BufferUtils.createFloatBuffer(16), shaders);
    }

    protected void bindAttribLocation(int program) {
    }

    public void updateMatrices() {
        glGetFloatv(GL_MODELVIEW_MATRIX, modelView);
        glGetFloatv(GL_PROJECTION_MATRIX, projection);
        updateMatrices = true;
    }

    public void use() {
        glUseProgram(name);

        if (updateMatrices) {
            glUniformMatrix4(modelViewUniform, false, modelView);
            glUniformMatrix4(projectionUniform, false, projection);
        }
    }

    public void end() {
        glUseProgram(0);
    }

    public abstract void drawArrays();

    public int getName() {
        return name;
    }

    public FloatBuffer getModelView() {
        return modelView;
    }

    public FloatBuffer getProjection() {
        return projection;
    }
}
