package tech.atani.client.util.game.render.font.gui.font.renderer.shader.sdf;

import tech.atani.client.util.game.render.font.gui.font.glyph.Glyph;
import tech.atani.client.util.game.render.font.gui.font.glyph.SdfAtlas;
import tech.atani.client.util.game.render.font.gui.shader.bo.BufferObject;
import tech.atani.client.util.game.render.font.gui.shader.vao.VertexArrayObject;
import tech.atani.client.util.game.render.font.gui.shader.vao.VertexAttribute;
import org.lwjgl.BufferUtils;

import java.awt.geom.Rectangle2D;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.FloatBuffer;
import java.util.Arrays;
import java.util.List;

import static org.lwjgl.opengl.GL11.GL_FLOAT;
import static org.lwjgl.opengl.GL11.GL_UNSIGNED_BYTE;
import static org.lwjgl.opengl.GL15.GL_ARRAY_BUFFER;

public class SdfTextBuffer {
    private final SdfShaderProgram program;

    private int bufferSize;
    private int length = 0;

    private ByteBuffer vertexData;
    private final VertexArrayObject vao;
    private final BufferObject vbo;
    //private final BufferObject ebo;

    private static final int VERTICES_PER_GLYPH = 6;

    private static final int POSITION_ATTRIB_BYTES = 3 * Float.BYTES;
    private static final int TEX_COORDS_ATTRIB_BYTES = 2 * Float.BYTES;
    private static final int COLOR_ATTRIB_BYTES = 4 * Byte.BYTES;

    public SdfTextBuffer(SdfShaderProgram program) {
        this.program = program;

        vao = new VertexArrayObject();
        vbo = new BufferObject(GL_ARRAY_BUFFER);
    }

    public void resize(int bufferSize, int usage) {
        createBuffer(bufferSize);
        setVaoAttributes();

        vbo.bind();
        vbo.bufferData(vertexData.capacity(), usage);
        vbo.unbind();
    }

    public void use() {
        program.use();
        program.updateUniforms();
        vao.bind();
    }

    public void draw() {
        program.drawArrays();
    }

    public void finish() {
        vao.unbind();
        program.end();
    }

    public void clearBuffer() {
        vertexData.clear();
        length = 0;
    }

    public void uploadBuffer() {
        vertexData.flip();
        program.setCharacterCount(length);

        vbo.bind();
        vbo.bufferSubData(0, vertexData);
        vbo.unbind();
    }

    public void addCharacter(SdfAtlas sdfAtlas,
                             int index, char ch,
                             float x, float y, float z,
                             int color, float scale) {
        if (index >= bufferSize) {
            throw new BufferOverflowException();
        }

        Glyph glyph = sdfAtlas.getGlyph(ch);
        if (glyph == null) {
            System.err.println("Error: Glyph is null for character '" + ch + "' in addCharacter.");
            return;
        }

        Rectangle2D bounds = glyph.getGlyphBounds();
        if (bounds == null) {
            System.err.println("Error: Glyph bounds are null for character '" + ch + "' in addCharacter.");
            return;
        }

        Rectangle2D.Float uv = glyph.getTexCoords();
        if (uv == null) {
            System.err.println("Error: Texture coordinates are null for character '" + ch + "' in addCharacter.");
            return;
        }

        float scaledWidth = (float) bounds.getWidth() * scale;
        float scaledHeight = (float) bounds.getHeight() * scale;

        int bytesOffset = POSITION_ATTRIB_BYTES * VERTICES_PER_GLYPH * index;

        vertexData.position(0);
        FloatBuffer vertexDataFloat = vertexData.asFloatBuffer();
        vertexDataFloat.position(bytesOffset / Float.BYTES);

        vertexDataFloat.put(new float[]{
                x, y, z,
                x + scaledWidth, y, z,
                x, y + scaledHeight, z,

                x, y + scaledHeight, z,
                x + scaledWidth, y + scaledHeight, z,
                x + scaledWidth, y, z
        });

        bytesOffset = POSITION_ATTRIB_BYTES * VERTICES_PER_GLYPH * bufferSize
                + TEX_COORDS_ATTRIB_BYTES * VERTICES_PER_GLYPH * index;
        vertexDataFloat.position(bytesOffset / Float.BYTES);

        vertexDataFloat.put(new float[]{
                uv.x, uv.y,
                uv.x + uv.width, uv.y,
                uv.x, uv.y + uv.height,

                uv.x, uv.y + uv.height,
                uv.x + uv.width, uv.y + uv.height,
                uv.x + uv.width, uv.y
        });

        bytesOffset = POSITION_ATTRIB_BYTES * VERTICES_PER_GLYPH * bufferSize
                + TEX_COORDS_ATTRIB_BYTES * VERTICES_PER_GLYPH * bufferSize
                + COLOR_ATTRIB_BYTES * VERTICES_PER_GLYPH * index;
        vertexData.position(bytesOffset);

        byte red = (byte) (color >> 16);
        byte green = (byte) (color >> 8);
        byte blue = (byte) (color >> 0);
        byte alpha = (byte) (color >> 24);

        vertexData.put(new byte[]{
                red, green, blue, alpha,
                red, green, blue, alpha,
                red, green, blue, alpha,

                red, green, blue, alpha,
                red, green, blue, alpha,
                red, green, blue, alpha
        });

        length++;
    }

    public int getBufferSize() {
        return bufferSize;
    }

    public SdfShaderProgram getProgram() {
        return program;
    }

    private void createBuffer(int bufferSize) {
        if (vertexData != null) {
            vertexData.clear();
        }

        this.bufferSize = bufferSize;

        int vboSize = (POSITION_ATTRIB_BYTES + TEX_COORDS_ATTRIB_BYTES + COLOR_ATTRIB_BYTES)
                * VERTICES_PER_GLYPH * bufferSize;
        vertexData = BufferUtils.createByteBuffer(vboSize);
    }

    private void setVaoAttributes() {
        List<VertexAttribute> attributes = Arrays.asList(
                new VertexAttribute(program, "in_position", 3, GL_FLOAT, false, POSITION_ATTRIB_BYTES, 0),
                new VertexAttribute(program, "in_texCoords", 2, GL_FLOAT, false, TEX_COORDS_ATTRIB_BYTES,
                        (long) POSITION_ATTRIB_BYTES * VERTICES_PER_GLYPH * bufferSize),
                new VertexAttribute(program, "in_color", 4, GL_UNSIGNED_BYTE, true, COLOR_ATTRIB_BYTES,
                        (long) POSITION_ATTRIB_BYTES * VERTICES_PER_GLYPH * bufferSize
                                + (long) TEX_COORDS_ATTRIB_BYTES * VERTICES_PER_GLYPH * bufferSize)
        );
        vao.setAttributes(vbo, null, attributes);
    }
}
