package tech.atani.client.util.game.render.font.gui.font.renderer.shader.sdf;

import lombok.Setter;
import tech.atani.client.util.game.render.font.gui.font.glyph.SdfAtlas;
import tech.atani.client.util.game.render.font.gui.shader.Shader;
import tech.atani.client.util.game.render.font.gui.shader.ShaderProgram;
import tech.atani.client.util.game.render.font.gui.shader.texture.Texture2d;

import java.awt.Color;

import static org.lwjgl.opengl.GL11.*;
import static org.lwjgl.opengl.GL13.GL_TEXTURE0;
import static org.lwjgl.opengl.GL13.glActiveTexture;
import static org.lwjgl.opengl.GL20.*;
import static org.lwjgl.opengl.GL30.GL_R8;

public class SdfShaderProgram extends ShaderProgram {
    private float threshold;
    private float smoothing;
    private float offset;

    private final Texture2d texture;

    private final int thresholdUniform;
    private final int smoothingUniform;
    private final int textBoundsMinUniform;
    private final int textBoundsMaxUniform;
    private final int gradientStartColorUniform;
    private final int gradientEndColorUniform;
    private final int tiltFactorUniform;
    private final int useGradientUniform;
    private final int offsetUniform;

    private boolean updateThreshold = false;
    private boolean updateSmoothing = false;
    private boolean updateTextBounds = false;
    private boolean updateGradientColors = false;
    private boolean updateTiltFactor = false;
    private boolean updateUseGradient = false;
    private boolean updateOffset = false;

    private float textBoundsMinX, textBoundsMinY;
    private float textBoundsMaxX, textBoundsMaxY;
    private Color gradientStartColor, gradientEndColor;
    private float tiltFactor;
    private int useGradient;

    @Setter
    private int characterCount;

    public SdfShaderProgram(String fragShaderSource, String vertShaderSource) {
        super(new Shader(fragShaderSource, GL_FRAGMENT_SHADER), new Shader(vertShaderSource, GL_VERTEX_SHADER));

        texture = new Texture2d(GL_TEXTURE_2D, 0, GL_R8, GL_RED, GL_UNSIGNED_BYTE);
        texture.bind();
        texture.texParameteri(GL_TEXTURE_MIN_FILTER, GL_LINEAR);
        texture.texParameteri(GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
        texture.unbind();

        thresholdUniform = glGetUniformLocation(name, "u_threshold");
        smoothingUniform = glGetUniformLocation(name, "u_smoothing");
        textBoundsMinUniform = glGetUniformLocation(name, "u_textBoundsMin");
        textBoundsMaxUniform = glGetUniformLocation(name, "u_textBoundsMax");
        gradientStartColorUniform = glGetUniformLocation(name, "u_gradientStartColor");
        gradientEndColorUniform = glGetUniformLocation(name, "u_gradientEndColor");
        tiltFactorUniform = glGetUniformLocation(name, "u_tiltFactor");
        useGradientUniform = glGetUniformLocation(name, "u_useGradient");
        offsetUniform = glGetUniformLocation(name, "u_offset");

        int textureUniform = glGetUniformLocation(name, "u_texture");
        use();
        glUniform1i(textureUniform, 0);
        end();
    }

    @Override
    public void drawArrays() {
        glActiveTexture(GL_TEXTURE0);
        texture.bind();

        glDrawArrays(GL_TRIANGLES, 0, 6 * characterCount);

        texture.unbind();
    }

    public void updateUniforms() {
        if (updateThreshold) {
            glUniform1f(thresholdUniform, threshold);
            updateThreshold = false;
        }

        if (updateSmoothing) {
            glUniform1f(smoothingUniform, smoothing);
            updateSmoothing = false;
        }

        if (updateTextBounds) {
            glUniform2f(textBoundsMinUniform, textBoundsMinX, textBoundsMinY);
            glUniform2f(textBoundsMaxUniform, textBoundsMaxX, textBoundsMaxY);
            updateTextBounds = false;
        }

        if (updateGradientColors) {
            glUniform4f(gradientStartColorUniform,
                    gradientStartColor.getRed() / 255.0f, gradientStartColor.getGreen() / 255.0f,
                    gradientStartColor.getBlue() / 255.0f, gradientStartColor.getAlpha() / 255.0f);
            glUniform4f(gradientEndColorUniform,
                    gradientEndColor.getRed() / 255.0f, gradientEndColor.getGreen() / 255.0f,
                    gradientEndColor.getBlue() / 255.0f, gradientEndColor.getAlpha() / 255.0f);
            updateGradientColors = false;
        }

        if (updateTiltFactor) {
            glUniform1f(tiltFactorUniform, tiltFactor);
            updateTiltFactor = false;
        }

        if (updateUseGradient) {
            glUniform1i(useGradientUniform, useGradient);
            updateUseGradient = false;
        }

        if (updateOffset) {
            glUniform1f(offsetUniform, offset);
            updateOffset = false;
        }
    }

    public void setThreshold(float threshold) {
        threshold = Math.max(0, Math.min(1, threshold));
        if (this.threshold == threshold) {
            return;
        }

        this.threshold = threshold;
        updateThreshold = true;
    }

    public void setSmoothing(float smoothing) {
        smoothing = Math.max(0, Math.min(1, smoothing));
        if (this.smoothing == smoothing) {
            return;
        }

        this.smoothing = smoothing;
        updateSmoothing = true;
    }

    public void setTextBounds(float minX, float minY, float maxX, float maxY) {
        if (this.textBoundsMinX == minX && this.textBoundsMinY == minY &&
                this.textBoundsMaxX == maxX && this.textBoundsMaxY == maxY) {
            return;
        }

        this.textBoundsMinX = minX;
        this.textBoundsMinY = minY;
        this.textBoundsMaxX = maxX;
        this.textBoundsMaxY = maxY;
        updateTextBounds = true;
    }

    public void setGradientColors(Color startColor, Color endColor) {
        if (this.gradientStartColor != null && this.gradientStartColor.equals(startColor) &&
                this.gradientEndColor != null && this.gradientEndColor.equals(endColor)) {
            return;
        }

        this.gradientStartColor = startColor;
        this.gradientEndColor = endColor;
        updateGradientColors = true;
    }

    public void setTiltFactor(float tiltFactor) {
        if (this.tiltFactor == tiltFactor) {
            return;
        }

        this.tiltFactor = tiltFactor;
        updateTiltFactor = true;
    }

    public void setUseGradient(boolean useGradient) {
        int value = useGradient ? 1 : 0;
        if (this.useGradient == value) {
            return;
        }

        this.useGradient = value;
        updateUseGradient = true;
    }

    public void setOffset(float offset) {
        if (this.offset == offset) {
            return;
        }

        this.offset = offset;
        updateOffset = true;
    }

    public void setTextureData(SdfAtlas sdfAtlas) {
        texture.bind();
        texture.texImage2d(sdfAtlas.getSize(), sdfAtlas.getSize(), sdfAtlas.getR8TextureBuffer());
        texture.unbind();
    }
}