package tech.atani.client.util.game.render.font.gui.font.renderer.shader.line;

import tech.atani.client.util.game.render.font.gui.font.ScaledFont;
import tech.atani.client.util.game.render.font.gui.shader.bo.BufferObject;
import tech.atani.client.util.game.render.font.gui.shader.vao.VertexArrayObject;
import tech.atani.client.util.game.render.font.gui.shader.vao.VertexAttribute;
import org.lwjgl.BufferUtils;

import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.FloatBuffer;
import java.util.Arrays;
import java.util.List;

import static org.lwjgl.opengl.GL11.GL_FLOAT;
import static org.lwjgl.opengl.GL11.GL_UNSIGNED_BYTE;
import static org.lwjgl.opengl.GL15.GL_ARRAY_BUFFER;

public class LineBuffer {
    private final LineShaderProgram program;

    private int bufferSize;
    private int count = 0;

    private ByteBuffer vertexData;
    private final VertexArrayObject vao;
    private final BufferObject vbo;

    private static final int TRIANGLE_VERTICES_COUNT = 3;
    private static final int TRIANGLES_PER_LINE = 2;

    private static final int POSITION_ATTRIB_BYTES = 3 * Float.BYTES;
    private static final int COLOR_ATTRIB_BYTES = 4 * Byte.BYTES;

    public LineBuffer(LineShaderProgram program) {
        this.program = program;

        vao = new VertexArrayObject();
        vbo = new BufferObject(GL_ARRAY_BUFFER);
    }

    public void resize(int bufferSize, int usage) {
        createBuffer(bufferSize);
        setVaoAttributes();

        vbo.bind();
        vbo.bufferData(vertexData.capacity(), usage);
        vbo.unbind();
    }

    public void use() {
        program.use();
        vao.bind();
    }

    public void draw() {
        program.drawArrays();
    }

    public void finish() {
        vao.unbind();
        program.end();
    }

    public void clearBuffer() {
        vertexData.clear();
        count = 0;
    }

    public void uploadBuffer() {
        vertexData.flip();
        program.setLineCount(count);

        vbo.bind();
        vbo.bufferSubData(0, vertexData);
        vbo.unbind();
    }

    public void addUnderline(int index,
                             float x, float y, float z,
                             ScaledFont font,
                             float charAscent, float charWidth,
                             int color) {
        addLine(index, x, y + charAscent + 0.1F * font.getSize() - font.getLineThickness() / 2F, z, charWidth, font.getLineThickness(), color);
    }

    public void addStrikethrough(int index,
                                 float x, float y, float z,
                                 ScaledFont font,
                                 float charAscent, float charWidth,
                                 int color) {
        addLine(index, x, y + charAscent - font.getFontHeight() / 5F - font.getLineThickness() / 2F, z, charWidth, font.getLineThickness(), color);
    }

    private void addLine(int index,
                         float x, float y, float z,
                         float width, float height,
                         int color) {
        if (index >= bufferSize) {
            throw new BufferOverflowException();
        }

        int bytesOffset = POSITION_ATTRIB_BYTES * TRIANGLE_VERTICES_COUNT * TRIANGLES_PER_LINE * index;

        vertexData.position(bytesOffset);
        FloatBuffer vertexDataFloat = vertexData.asFloatBuffer();

        vertexDataFloat.put(new float[]{
                // first triangle positions
                x,              y,              z,
                x + width,      y,              z,
                x,              y + height,     z,

                // second triangle positions
                x,              y + height,     z,
                x + width,      y + height,     z,
                x + width,      y,              z
        });

        bytesOffset = POSITION_ATTRIB_BYTES * TRIANGLE_VERTICES_COUNT * TRIANGLES_PER_LINE * bufferSize
                + COLOR_ATTRIB_BYTES * TRIANGLE_VERTICES_COUNT * TRIANGLES_PER_LINE * index;
        vertexData.position(bytesOffset);

        byte red    = (byte) (color >> 16);
        byte green  = (byte) (color >> 8);
        byte blue   = (byte) (color >> 0);
        byte alpha  = (byte) (color >> 24);

        vertexData.put(new byte[] {
                // first triangle colors
                red, green, blue, alpha,
                red, green, blue, alpha,
                red, green, blue, alpha,

                // second triangle colors
                red, green, blue, alpha,
                red, green, blue, alpha,
                red, green, blue, alpha
        });

        count++;
    }

    public int getBufferSize() {
        return bufferSize;
    }

    private void createBuffer(int bufferSize) {
        if (vertexData != null) {
            vertexData.clear();
        }

        bufferSize *= 2;
        this.bufferSize = bufferSize;

        int vboSize = (POSITION_ATTRIB_BYTES + COLOR_ATTRIB_BYTES)
                * TRIANGLE_VERTICES_COUNT * TRIANGLES_PER_LINE * bufferSize;
        vertexData = BufferUtils.createByteBuffer(vboSize);
    }

    private void setVaoAttributes() {
        List<VertexAttribute> attributes = Arrays.asList(
                new VertexAttribute(program, "in_position", 3, GL_FLOAT, false, POSITION_ATTRIB_BYTES, 0),
                new VertexAttribute(program, "in_color", 4, GL_UNSIGNED_BYTE, true, COLOR_ATTRIB_BYTES,
                        (long) POSITION_ATTRIB_BYTES * TRIANGLE_VERTICES_COUNT * TRIANGLES_PER_LINE * bufferSize)
        );
        vao.setAttributes(vbo, null, attributes);
    }
}
