package tech.atani.client.util.game.render.font.gui.font.renderer;

import com.ibm.icu.text.ArabicShaping;
import com.ibm.icu.text.ArabicShapingException;
import com.ibm.icu.text.Bidi;
import tech.atani.client.util.game.render.font.gui.font.ScaledFont;
import tech.atani.client.util.game.render.font.gui.font.formatting.FormattingProcessor;
import tech.atani.client.util.game.render.font.gui.font.formatting.style.StyledCharacter;
import tech.atani.client.util.game.render.font.gui.font.renderer.shader.line.LineBuffer;
import tech.atani.client.util.game.render.font.gui.font.renderer.shader.sdf.SdfTextBuffer;

import java.awt.*;
import java.util.Arrays;
import java.util.List;

import static com.ibm.icu.text.Bidi.DIRECTION_DEFAULT_RIGHT_TO_LEFT;

public class SimpleFontRenderer {
    private final ScaledFont font;
    private final SdfTextBuffer textBuffer;
    private final LineBuffer lineBuffer;
    private final FormattingProcessor processor;

    private boolean bidiEnabled;

    public SimpleFontRenderer(ScaledFont font, SdfTextBuffer textBuffer, LineBuffer lineBuffer, FormattingProcessor processor) {
        this.font = font;
        this.textBuffer = textBuffer;
        this.lineBuffer = lineBuffer;
        this.processor = processor;
    }

    public int drawString(String text, float x, float y, int color, boolean dropShadow) {
        int i;

        if (dropShadow) {
            i = renderString(text, x + 1, y + 1, color, true, false, 0L, null, null);
            i = Math.max(i, renderString(text, x, y, color, false, false, 0L, null, null));
        } else {
            i = renderString(text, x, y, color, false, false, 0L, null, null);
        }

        return i;
    }

    public int drawString(String text, float x, float y, int color) {
        return drawString(text, x, y, color, false);
    }

    public int drawString(String text, float x, float y, Color color) {
        return drawString(text, x, y, color.getRGB());
    }

    public int drawXCenteredString(String text, float x, float y, int color, boolean dropShadow) {
        return drawString(text, x - processor.getStringWidth(font, text) / 2F, y, color, dropShadow);
    }

    public int drawXCenteredString(String text, float x, float y, Color color, boolean dropShadow) {
        return drawString(text, x - processor.getStringWidth(font, text) / 2F, y, color.getRGB(), dropShadow);
    }

    public int drawXCenteredString(String text, float x, float y, int color) {
        return drawXCenteredString(text, x, y, color, false);
    }

    public int drawXCenteredString(String text, float x, float y, Color color) {
        return drawXCenteredString(text, x, y, color.getRGB());
    }

    public int drawYCenteredString(String text, float x, float y, int color, boolean dropShadow) {
        return drawString(text, x, y - font.getFontHeight() / 2F, color, dropShadow);
    }

    public int drawYCenteredString(String text, float x, float y, Color color, boolean dropShadow) {
        return drawString(text, x, y - font.getFontHeight() / 2F, color.getRGB(), dropShadow);
    }

    public int drawYCenteredString(String text, float x, float y, int color) {
        return drawYCenteredString(text, x, y, color, false);
    }

    public int drawYCenteredString(String text, float x, float y, Color color) {
        return drawYCenteredString(text, x, y, color.getRGB());
    }

    public int drawXYCenteredString(String text, float x, float y, int color, boolean dropShadow) {
        return drawString(text, x - processor.getStringWidth(font, text) / 2F, y - font.getFontHeight() / 2F, color, dropShadow);
    }

    public int drawXYCenteredString(String text, float x, float y, Color color, boolean dropShadow) {
        return drawString(text, x - processor.getStringWidth(font, text) / 2F, y - font.getFontHeight() / 2F, color.getRGB(), dropShadow);
    }

    public int drawXYCenteredString(String text, float x, float y, int color) {
        return drawXYCenteredString(text, x, y, color, false);
    }

    public int drawXYCenteredString(String text, float x, float y, Color color) {
        return drawXYCenteredString(text, x, y, color.getRGB());
    }

    public int drawGradientString(String text, float x, float y, long offset, Color startColor, Color endColor, boolean dropShadow) {
        int i;

        if (dropShadow) {
            i = renderString(text, x + 0.8f, y + 0.8f, new Color(24, 24, 24).getRGB(), true, false, offset, null, null);
            i = Math.max(i, renderString(text, x, y, 0, false, true, offset, startColor, endColor));
        } else {
            i = renderString(text, x, y, 0, false, true, offset, startColor, endColor);
        }

        return i;
    }

    public int drawXCenteredGradientString(String text, float x, float y, long offset, Color startColor, Color endColor, boolean dropShadow) {
        float adjustedX = x - processor.getStringWidth(font, text) / 2F;
        return drawGradientString(text, adjustedX, y, offset, startColor, endColor, dropShadow);
    }

    public int drawXCenteredGradientString(String text, float x, float y, long offset, Color startColor, Color endColor) {
        return drawXCenteredGradientString(text, x, y, offset, startColor, endColor, false);
    }

    public int drawYCenteredGradientString(String text, float x, float y, long offset, Color startColor, Color endColor, boolean dropShadow) {
        float adjustedY = y - font.getFontHeight() / 2F;
        return drawGradientString(text, x, adjustedY, offset, startColor, endColor, dropShadow);
    }

    public int drawYCenteredGradientString(String text, float x, float y, long offset, Color startColor, Color endColor) {
        return drawYCenteredGradientString(text, x, y, offset, startColor, endColor, false);
    }

    public int drawXYCenteredGradientString(String text, float x, float y, long offset, Color startColor, Color endColor, boolean dropShadow) {
        float adjustedX = x - processor.getStringWidth(font, text) / 2F;
        float adjustedY = y - font.getFontHeight() / 2F;
        return drawGradientString(text, adjustedX, adjustedY, offset, startColor, endColor, dropShadow);
    }

    public int drawXYCenteredGradientString(String text, float x, float y, long offset, Color startColor, Color endColor) {
        return drawXYCenteredGradientString(text, x, y, offset, startColor, endColor, false);
    }

    public int getStringWidth(String text) {
        return Math.round(processor.getStringWidth(font, text));
    }

    public float getStringWidthPure(String text) {
        return processor.getStringWidth(font, text);
    }

    public int getFontHeight() {
        return Math.round(font.getFontHeight());
    }

    public int getCharWidth(char ch) {
        return Math.round(font.getCharWidth(ch));
    }

    public int getCharAscent(char ch) {
        return Math.round(font.getCharAscent(ch));
    }

    public String trimStringToWidth(String text, int width) {
        return processor.trimStringToWidth(font, text, width);
    }

    public String trimStringToWidth(String text, int width, boolean reverse) {
        return processor.trimStringToWidth(font, text, width, reverse);
    }

    public void drawSplitString(String str, float x, float y, float wrapWidth, int color, boolean dropShadow) {
        str = trimStringNewline(str);
        renderSplitString(str, x, y, wrapWidth, color, dropShadow);
    }

    public int splitStringWidth(String str, int maxLength) {
        return Math.round(font.getFontHeight() * listFormattedStringToWidth(str, maxLength).size());
    }

    public List<String> listFormattedStringToWidth(String str, float wrapWidth) {
        return Arrays.asList(wrapFormattedStringToWidth(str, wrapWidth).split("\n"));
    }

    public boolean isBidiEnabled() {
        return this.bidiEnabled;
    }

    public void setBidiEnabled(boolean bidiFlag) {
        this.bidiEnabled = bidiFlag;
    }

    private void renderStringAligned(String text, float x, float y, float width, int defaultColor, boolean shadow) {
        if (bidiEnabled) {
            int i = getStringWidth(bidiReorder(text));
            x = x + width - i;
        }

        renderString(text, x, y, defaultColor, shadow, false, 0L, null, null);
    }

    private int renderString(String text, float x, float y, int defaultColor, boolean shadow, boolean useGradient) {
        return renderString(text, x, y, defaultColor, shadow, useGradient, 0L, null, null);
    }

    private int renderString(String text, float x, float y, int defaultColor, boolean shadow, boolean useGradient, long offset, Color startColor, Color endColor) {
        if (font.getSize() == 0) {
            return (int) x;
        }

        if (bidiEnabled) {
            text = bidiReorder(text);
        }

        List<StyledCharacter> chars = processor.chars(text, defaultColor, shadow);

        float totalWidth = processor.getStringWidth(font, text);
        float totalHeight = font.getFontHeight();
        float minX = x;
        float minY = y;
        float maxX = x + totalWidth;
        float maxY = y + totalHeight;

        ChunkedRenderer renderer = new ChunkedRenderer(font, chars, textBuffer, lineBuffer, x, y, 0);

        renderer.begin();

        if (useGradient && !shadow) {
            Color gradientStart = (startColor != null) ? startColor : new Color(0xFC6A8C);
            Color gradientEnd = (endColor != null) ? endColor : new Color(0x6AD5FC);
            float tiltFactor = 0.2f;
            float gradientOffset = (offset % 2000L) / 2000.0f;

            textBuffer.getProgram().setTextBounds(minX, minY, maxX, maxY);
            textBuffer.getProgram().setGradientColors(gradientStart, gradientEnd);
            textBuffer.getProgram().setTiltFactor(tiltFactor);
            textBuffer.getProgram().setUseGradient(true);
            textBuffer.getProgram().setOffset(gradientOffset);
        } else {
            textBuffer.getProgram().setUseGradient(false);
        }

        while (renderer.hasNext()) {
            x += renderer.upload();
            renderer.render();
        }
        renderer.end();
        return (int) x;
    }

    private void renderSplitString(String str, float x, float y, float wrapWidth, int color, boolean dropShadow) {
        for (String s : listFormattedStringToWidth(str, wrapWidth)) {
            renderStringAligned(s, x, y, wrapWidth, color, dropShadow);
            y += (float) Math.ceil(font.getFontHeight());
        }
    }

    private String trimStringNewline(String text) {
        while (text != null && text.endsWith("\n")) {
            text = text.substring(0, text.length() - 1);
        }

        return text;
    }

    private String wrapFormattedStringToWidth(String str, float wrapWidth) {
        if (str.length() <= 1) {
            return str;
        }

        int i = processor.sizeStringToWidth(font, str, wrapWidth);

        if (str.length() <= i) {
            return str;
        }

        String s = str.substring(0, i);
        char ch = str.charAt(i);
        boolean flag = ch == ' ' || ch == '\n';
        String s1 = processor.getFormatFromString(s) + str.substring(i + (flag ? 1 : 0));
        return s + "\n" + wrapFormattedStringToWidth(s1, wrapWidth);
    }

    private String bidiReorder(String text) {
        try {
            Bidi bidi = new Bidi((new ArabicShaping(8)).shape(text), DIRECTION_DEFAULT_RIGHT_TO_LEFT);
            bidi.setReorderingMode(0);
            return bidi.writeReordered(2);
        } catch (ArabicShapingException e) {
            return text;
        }
    }
}