package tech.atani.client.util.game.render.font.gui.font.renderer;

import tech.atani.client.util.game.render.font.gui.font.Font;
import tech.atani.client.util.game.render.font.gui.font.ScaledFont;
import tech.atani.client.util.game.render.font.gui.font.formatting.style.StyledCharacter;
import tech.atani.client.util.game.render.font.gui.font.formatting.style.TextStyle;
import tech.atani.client.util.game.render.font.gui.font.glyph.SdfAtlas;
import tech.atani.client.util.game.render.font.gui.font.renderer.shader.line.LineBuffer;
import tech.atani.client.util.game.render.font.gui.font.renderer.shader.sdf.SdfShaderProgram;
import tech.atani.client.util.game.render.font.gui.font.renderer.shader.sdf.SdfTextBuffer;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.GlStateManager;

import java.nio.FloatBuffer;
import java.util.List;
import java.util.Random;

public class ChunkedRenderer {
    private static final float SMOOTHING_PER_SPREAD = 0.9F;
    private static final Random random = new Random();
    private final static String RANDOM_CHARS = "ÀÁÂÈÊËÍÓÔÕÚßãõğİıŒœŞşŴŵžȇ\u0000\u0000\u0000\u0000\u0000\u0000\u0000 !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\u0000ÇüéâäàåçêëèïîìÄÅÉæÆôöòûùÿÖÜø£Ø×ƒáíóúñÑªº¿®¬½¼¡«»░▒▓│┤╡╢╖╕╣║╗╝╜╛┐└┴┬├─┼╞╟╚╔╩╦╠═╬╧╨╤╥╙╘╒╓╫╪┘┌█▄▌▐▀αβΓπΣσμτΦΘΩδ∞∅∈∩≡±≥≤⌠⌡÷≈°∙·√ⁿ²■\u0000";

    private final ScaledFont font;
    private final List<StyledCharacter> chars;
    private final SdfAtlas sdfAtlas;

    private final SdfTextBuffer textBuffer;
    private final SdfShaderProgram program;

    private final LineBuffer lineBuffer;

    private final float x;
    private final float y;
    private final float z;

    private int index;

    public ChunkedRenderer(ScaledFont font, List<StyledCharacter> chars, SdfTextBuffer textBuffer, LineBuffer lineBuffer, float x, float y, float z) {
        this.font = font;
        this.chars = chars;
        this.sdfAtlas = font.getAtlas();
        this.textBuffer = textBuffer;
        this.program = textBuffer.getProgram();
        this.lineBuffer = lineBuffer;

        this.x = x;
        this.y = y;
        this.z = z;
    }

    public void begin() {
        index = 0;

        float threshold = Font.WEIGHT_REGULAR / font.getWeight() * 0.5F;

        program.updateMatrices();

        float smoothing = getSmoothing();

        program.setSmoothing(smoothing);
        program.setThreshold(threshold);

        GlStateManager.disableCull();
        GlStateManager.enableBlend();
        GlStateManager.enableAlpha();
    }

    private int lineIndex;

    public boolean hasNext() {
        return index < chars.size();
    }

    public float upload() {
        textBuffer.clearBuffer();
        lineBuffer.clearBuffer();

        float width = 0;
        lineIndex = 0;
        for (int glyphIndex = 0; index < chars.size() && glyphIndex < textBuffer.getBufferSize(); index++) {
            StyledCharacter styledCharacter = chars.get(index);

            char ch = styledCharacter.getCharacter();
            TextStyle style = styledCharacter.getStyle();

            float charWidth = font.getCharWidth(ch);

            if (style.isRandom()) {
                float newCharWidth;
                do {
                    int j = random.nextInt(RANDOM_CHARS.length());
                    ch = RANDOM_CHARS.charAt(j);
                    newCharWidth = font.getCharWidth(ch);
                } while (Math.abs(charWidth - newCharWidth) > 25);
            }

            textBuffer.addCharacter(sdfAtlas, glyphIndex, ch, x + width, y, z, style.getColor(), font.getScale());
            glyphIndex++;

            float charAscent = font.getCharAscent(ch);

            if (style.isUnderline()) {
                lineBuffer.addUnderline(lineIndex, x + width, y, z, font, charAscent, charWidth, style.getColor());
                lineIndex++;
            }
            if (style.isStrikethrough()) {
                lineBuffer.addStrikethrough(lineIndex, x + width, y, z, font, charAscent, charWidth, style.getColor());
                lineIndex++;
            }

            width += charWidth;
        }

        textBuffer.uploadBuffer();

        if (lineIndex > 0) {
            lineBuffer.uploadBuffer();
        }
        return width;
    }

    public void render() {
        textBuffer.use();
        textBuffer.draw();

        if (lineIndex > 0) {
            lineBuffer.use();
            lineBuffer.draw();
        }
    }

    public void end() {
        textBuffer.finish();
    }

    /**
     * Gives the smoothing factor for the shader based on the scale of the font and the scale of the game
     */
    private float getSmoothing() {
        Minecraft mc = Minecraft.getMinecraft();

        FloatBuffer projection = program.getProjection();
        float m00 = projection.get(0);
        float m11 = projection.get(1 + 4);

        float m03 = projection.get(3 * 4);
        float m13 = projection.get(1 + 3 * 4);

        float scaleX = m00 * mc.displayWidth + m03;
        float scaleY = m11 * -mc.displayHeight - m13;
        float currentScale = (float) Math.sqrt(scaleX * scaleY);

        float spread = sdfAtlas.getSpread();
        return Math.abs(SMOOTHING_PER_SPREAD / spread / font.getScale() / currentScale);
    }
}
