package tech.atani.client.util.game.render.font.gui.font.glyph;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;
import org.lwjgl.BufferUtils;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.font.FontRenderContext;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.DataBuffer;
import java.io.*;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import static java.awt.RenderingHints.*;
import static java.awt.image.BufferedImage.TYPE_BYTE_GRAY;

public class SdfAtlas {
    public static final char MISSING_CHARACTER = '□';
    public static final char INVALID_CHARACTER = '�';

    private final UUID uuid;
    private final float fontSize;
    private final double maxGlyphHeight;
    private final Map<Character, Glyph> glyphs;

    private final int size;
    private final int spread;

    private final ByteBuffer r8TextureBuffer;

    private final BufferedImage image;

    private SdfAtlas(UUID uuid, float fontSize, double maxGlyphHeight, Map<Character, Glyph> glyphs,
                     int size, int spread, ByteBuffer r8TextureBuffer, BufferedImage image) {
        this.uuid = uuid;
        this.fontSize = fontSize;
        this.maxGlyphHeight = maxGlyphHeight;
        this.glyphs = glyphs;

        this.size = size;
        this.spread = spread;
        this.r8TextureBuffer = r8TextureBuffer;

        this.image = image;
    }

    public float getFontSize() {
        return fontSize;
    }

    public double getMaxGlyphHeight() {
        return maxGlyphHeight;
    }

    public Glyph getGlyph(char ch) {
        Glyph glyph = glyphs.get(ch);
        if (glyph == null) {
            if (Character.isDefined(ch)) {
                glyph = glyphs.get(MISSING_CHARACTER);
            } else {
                glyph = glyphs.get(INVALID_CHARACTER);
            }
        }
        return glyph;
    }

    public int getSize() {
        return size;
    }

    public int getSpread() {
        return spread;
    }

    public ByteBuffer getR8TextureBuffer() {
        return r8TextureBuffer;
    }

    public BufferedImage getImage() {
        return image;
    }

    public static SdfAtlas fromFont(Font font, char[] characters, int spread, int downscale) {
        long time = System.currentTimeMillis();


        FontRenderContext frc = new FontRenderContext(new AffineTransform(), true, false);

        Rectangle2D bounds = font.getStringBounds(characters, 0, characters.length, frc);
        double maxGlyphHeight = bounds.getHeight();

        int totalWidth = (int) Math.ceil(bounds.getWidth()) + (spread * 2) * characters.length;
        int rowHeight = (int) Math.ceil(maxGlyphHeight) + (spread * 2);

        int sizeIn = (int) Math.ceil((
                Math.sqrt(totalWidth * rowHeight) + rowHeight / 2.0
        ) / 2.0) * 2; // round to even number

        Map<Character, Glyph> glyphs = new HashMap<>(characters.length);

        BufferedImage image = new BufferedImage(sizeIn, sizeIn, TYPE_BYTE_GRAY);

        Graphics2D graphics = image.createGraphics();
        graphics.setRenderingHint(KEY_TEXT_ANTIALIASING, VALUE_TEXT_ANTIALIAS_ON);
        graphics.setRenderingHint(KEY_FRACTIONALMETRICS, VALUE_FRACTIONALMETRICS_OFF);

        graphics.setFont(font);
        graphics.setColor(Color.WHITE);

        FontMetrics fontMetrics = graphics.getFontMetrics();

        int currentX = spread;
        int currentY = spread;
        char[] data = new char[1];
        for (char ch : characters) {
            data[0] = ch;

            Rectangle2D glyphBounds = fontMetrics.getStringBounds(data, 0, data.length, graphics);
            if (currentX + glyphBounds.getWidth() + spread > sizeIn) {
                // go to the beginning of the next line
                currentX = spread;
                currentY += rowHeight;
            }

            graphics.drawChars(data, 0, data.length, currentX, currentY - (int) glyphBounds.getY());

            Rectangle2D.Float uv = new Rectangle2D.Float(
                    currentX / (float) sizeIn,
                    currentY / (float) sizeIn,
                    (float) glyphBounds.getWidth() / (float) sizeIn,
                    (float) glyphBounds.getHeight() / (float) sizeIn
            );
            glyphs.put(ch, new Glyph(ch, glyphBounds, uv));

            currentX += (int) Math.ceil(glyphBounds.getWidth()) + spread * 2;
        }

        DataBuffer dataBufferIn = image.getRaster().getDataBuffer();

        int sizeOut = sizeIn / downscale;

        boolean[][] pixelsIn = new boolean[sizeIn][sizeIn];

        for (int y = 0; y < sizeIn; y++) {
            for (int x = 0; x < sizeIn; x++) {
                pixelsIn[y][x] = isInside(dataBufferIn.getElem(y * sizeIn + x));
            }
        }

        BufferedImage imageOut = new BufferedImage(sizeOut, sizeOut, TYPE_BYTE_GRAY);
        DataBuffer dataBufferOut = imageOut.getRaster().getDataBuffer();
        ByteBuffer r8TextureBuffer = BufferUtils.createByteBuffer(sizeOut * sizeOut);

        for (int y = 0; y < sizeOut; y++) {
            for (int x = 0; x < sizeOut; x++) {
                int xIn = x * downscale + downscale / 2;
                int yIn = y * downscale + downscale / 2;
                boolean inside = pixelsIn[yIn][xIn];

                float closestDist = findClosestOppositeColorDistance(pixelsIn, xIn, yIn, sizeIn, inside, spread);
                float mappedDist = closestDist / (float) spread;

                int alpha = (int) ((0.5F + 0.5F * mappedDist * (inside ? 1 : -1)) * 0xFF);
                dataBufferOut.setElem(y * sizeOut + x, alpha);
                r8TextureBuffer.put((byte) alpha);
            }
        }

        r8TextureBuffer.flip();

        return new SdfAtlas(UUID.randomUUID(), font.getSize2D(), maxGlyphHeight, glyphs, sizeOut, spread, r8TextureBuffer, imageOut);
    }

    private static boolean isInside(final int argb) {
        return (argb & 0x80) != 0;
    }

    private static float findClosestOppositeColorDistance(boolean[][] pixels, int x, int y, int sizeIn, boolean inside, int spread) {
        int closestSquaredDist = spread * spread;
        int startX = Math.max(0, x - spread);
        int endX = Math.min(sizeIn, x + spread);
        int startY = Math.max(0, y - spread);
        int endY = Math.min(sizeIn, y + spread);


        for (int sy = startY; sy < endY; sy++) {
            for (int sx = startX; sx < endX; sx++) {
                if (pixels[sy][sx] == inside) {
                    continue;
                }

                int squaredDist = squaredDistance(x, y, sx, sy);
                if (squaredDist < closestSquaredDist) {
                    closestSquaredDist = squaredDist;
                }
            }
        }

        return (float) Math.sqrt(closestSquaredDist);
    }

    private static int squaredDistance(int x, int y, int x2, int y2) {
        int xDiff = x2 - x;
        int yDiff = y2 - y;
        return xDiff * xDiff + yDiff * yDiff;
    }

    public void write(JsonWriter writer, Path atlasesDirectory) throws IOException {
        writer.beginObject();

        writer.name("uuid").value(uuid.toString());
        writer.name("fontSize").value(fontSize);
        writer.name("maxGlyphHeight").value(maxGlyphHeight);

        writer.name("size").value(size);
        writer.name("spread").value(spread);

        writer.name("glyphs").beginArray();
        for (Entry<Character, Glyph> entry : glyphs.entrySet()) {
            entry.getValue().write(writer);
        }
        writer.endArray();

        writer.endObject();

        Path imagePath = atlasesDirectory.resolve(uuid.toString().concat(".png"));
        if (Files.isRegularFile(imagePath)) {
            return;
        }

        try (OutputStream stream = new BufferedOutputStream(Files.newOutputStream(imagePath))) {
            ImageIO.write(image, "png", stream);
        }
    }

    public static SdfAtlas read(JsonReader reader, Path atlasesDirectory) throws IOException {
        JsonObject object = new JsonParser().parse(reader).getAsJsonObject();

        JsonElement uuidElement = object.get("uuid");
        JsonElement fontSizeElement = object.get("fontSize");
        JsonElement maxGlyphHeightElement = object.get("maxGlyphHeight");
        JsonElement sizeElement = object.get("size");
        JsonElement spreadElement = object.get("spread");
        JsonArray glyphsElement = object.getAsJsonArray("glyphs");

        if (uuidElement == null || fontSizeElement == null || maxGlyphHeightElement == null || sizeElement == null || spreadElement == null || glyphsElement == null) {
            throw new InvalidObjectException("missing element");
        }

        UUID uuid = UUID.fromString(uuidElement.getAsString());
        float fontSize = fontSizeElement.getAsFloat();
        double maxGlyphHeight = maxGlyphHeightElement.getAsDouble();
        int size = sizeElement.getAsInt();
        int spread = spreadElement.getAsInt();

        Map<Character, Glyph> glyphs = new HashMap<>();
        for (JsonElement glyphElement : glyphsElement) {
            JsonObject glyphObject = glyphElement.getAsJsonObject();
            Glyph glyph = Glyph.read(glyphObject);
            glyphs.put(glyph.getCharacter(), glyph);
        }

        Path imagePath = atlasesDirectory.resolve(uuid.toString().concat(".png"));
        if (!Files.isRegularFile(imagePath)) {
            throw new FileNotFoundException("atlas image missing: " + imagePath);
        }

        BufferedImage image;
        try (InputStream stream = new BufferedInputStream(Files.newInputStream(imagePath))) {
            image = ImageIO.read(stream);
        }

        long time = System.currentTimeMillis();


        if (size != image.getWidth() || size != image.getHeight()) {
            throw new IllegalArgumentException("image isn't of the expected size: "
                    + image.getWidth() + "x" + image.getHeight() + " instead of " + size + "x" + size);
        }

        DataBuffer dataBufferIn = image.getRaster().getDataBuffer();

        ByteBuffer r8TextureBuffer = BufferUtils.createByteBuffer(size * size);

        for (int y = 0; y < size; y++) {
            for (int x = 0; x < size; x++) {
                r8TextureBuffer.put((byte) dataBufferIn.getElem(y * size + x));
            }
        }

        r8TextureBuffer.flip();

        return new SdfAtlas(uuid, fontSize, maxGlyphHeight, glyphs, size, spread, r8TextureBuffer, image);
    }
}
