package tech.atani.client.util.game.render.font.gui.font.glyph;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.stream.JsonWriter;

import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.InvalidObjectException;

public class Glyph {
    private final char character;
    private final Rectangle2D glyphBounds;
    private final Rectangle2D.Float texCoords;

    public Glyph(char character, Rectangle2D glyphBounds, Rectangle2D.Float texCoords) {
        this.character = character;
        this.glyphBounds = glyphBounds;
        this.texCoords = texCoords;
    }

    public char getCharacter() {
        return character;
    }

    public Rectangle2D getGlyphBounds() {
        return glyphBounds;
    }

    public Rectangle2D.Float getTexCoords() {
        return texCoords;
    }

    public void write(JsonWriter writer) throws IOException {
        writer.beginObject();

        writer.name("char").value(Character.toString(character));

        writer.name("bounds").beginObject();
        writer.name("x").value(glyphBounds.getX());
        writer.name("y").value(glyphBounds.getY());
        writer.name("w").value(glyphBounds.getWidth());
        writer.name("h").value(glyphBounds.getHeight());
        writer.endObject();

        writer.name("uv").beginObject();
        writer.name("x").value(texCoords.x);
        writer.name("y").value(texCoords.y);
        writer.name("w").value(texCoords.width);
        writer.name("h").value(texCoords.height);
        writer.endObject();

        writer.endObject();
    }

    public static Glyph read(JsonObject object) throws IOException {
        JsonElement charElement = object.get("char");
        JsonObject boundsElement = object.getAsJsonObject("bounds");
        JsonObject uvElement = object.getAsJsonObject("uv");

        if (charElement == null || boundsElement == null || uvElement == null) {
            throw new InvalidObjectException("missing element");
        }

        String charString = charElement.getAsString();
        if (charString.isEmpty()) {
            throw new InvalidObjectException("char is empty");
        }

        Rectangle2D bounds = readBounds(boundsElement);
        Rectangle2D.Float texCoords = readTexCoords(uvElement);

        return new Glyph(charString.charAt(0), bounds, texCoords);
    }

    private static Rectangle2D readBounds(JsonObject object) throws IOException {
        JsonElement xElement = object.get("x");
        JsonElement yElement = object.get("y");
        JsonElement wElement = object.get("w");
        JsonElement hElement = object.get("h");

        if (xElement == null || yElement == null || wElement == null || hElement == null) {
            throw new InvalidObjectException("missing element");
        }

        return new Rectangle2D.Double(xElement.getAsDouble(), yElement.getAsDouble(), wElement.getAsDouble(), hElement.getAsDouble());
    }

    private static Rectangle2D.Float readTexCoords(JsonObject object) throws IOException {
        JsonElement xElement = object.get("x");
        JsonElement yElement = object.get("y");
        JsonElement wElement = object.get("w");
        JsonElement hElement = object.get("h");

        if (xElement == null || yElement == null || wElement == null || hElement == null) {
            throw new InvalidObjectException("missing element");
        }

        return new Rectangle2D.Float(xElement.getAsFloat(), yElement.getAsFloat(), wElement.getAsFloat(), hElement.getAsFloat());
    }
}