package tech.atani.client.util.game.render.font.gui.font;

import tech.atani.client.util.game.render.font.gui.font.formatting.FormattingProcessor;
import tech.atani.client.util.game.render.font.gui.font.glyph.SdfAtlas;
import tech.atani.client.util.game.render.font.gui.font.renderer.SimpleFontRenderer;

import java.util.HashSet;
import java.util.Set;

public class ScaledFont {
    private final Font font;
    private final String name;
    private final SdfAtlas atlas;

    private final float size;
    private final float weight;
    private final float maxGlyphHeight;

    private final float scale;
    private final float lineThickness;

    private static final Set<String> loggedErrors = new HashSet<>();

    public ScaledFont(Font font, float size, float weight) {
        if (font == null) {
            logError("Font is null in ScaledFont constructor.");
            throw new IllegalArgumentException("Font cannot be null.");
        }
        if (font.getAtlas() == null) {
            logError("Font atlas is null in ScaledFont constructor.");
            throw new IllegalArgumentException("Font atlas cannot be null.");
        }

        this.font = font;
        this.name = font.getName();
        this.atlas = font.getAtlas();
        this.size = size;
        this.weight = weight;
        this.scale = size / font.getAtlas().getFontSize();
        this.maxGlyphHeight = (float) atlas.getMaxGlyphHeight() * scale;
        this.lineThickness = Math.max(0.06F * size, 1);
    }

    public SimpleFontRenderer newRenderer() {
        return newRenderer(FormattingProcessor.createDefault());
    }

    public SimpleFontRenderer newRenderer(FormattingProcessor processor) {
        if (processor == null) {
            logError("FormattingProcessor is null in newRenderer.");
            return null;
        }
        return new SimpleFontRenderer(this, font.getTextBuffer(), font.getLineBuffer(), processor);
    }

    public String getName() {
        return name;
    }

    public SdfAtlas getAtlas() {
        return atlas;
    }

    public float getSize() {
        return size;
    }

    public float getWeight() {
        return weight;
    }

    public float getScale() {
        return scale;
    }

    public float getFontHeight() {
        return maxGlyphHeight;
    }

    public float getCharWidth(char ch) {
        if (atlas == null) {
            logError("Atlas is null in getCharWidth.");
            return 0;
        }
        var glyph = atlas.getGlyph(ch);
        if (glyph == null) {
            logError("Glyph is null for character '" + ch + "' in getCharWidth.");
            return 0;
        }
        if (glyph.getGlyphBounds() == null) {
            logError("Glyph bounds are null for character '" + ch + "' in getCharWidth.");
            return 0;
        }
        return (float) glyph.getGlyphBounds().getWidth() * scale;
    }

    public float getCharAscent(char ch) {
        if (atlas == null) {
            logError("Atlas is null in getCharAscent.");
            return 0;
        }
        var glyph = atlas.getGlyph(ch);
        if (glyph == null) {
            logError("Glyph is null for character '" + ch + "' in getCharAscent.");
            return 0;
        }
        if (glyph.getGlyphBounds() == null) {
            logError("Glyph bounds are null for character '" + ch + "' in getCharAscent.");
            return 0;
        }
        return (float) -glyph.getGlyphBounds().getY() * scale;
    }

    public float getLineThickness() {
        return lineThickness;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ScaledFont that = (ScaledFont) o;

        if (Float.compare(size, that.size) != 0) return false;
        if (Float.compare(weight, that.weight) != 0) return false;
        return atlas.equals(that.atlas);
    }

    @Override
    public int hashCode() {
        int result = atlas.hashCode();
        result = 31 * result + (size != 0.0f ? Float.floatToIntBits(size) : 0);
        result = 31 * result + (weight != 0.0f ? Float.floatToIntBits(weight) : 0);
        return result;
    }

    private static void logError(String message) {
        if (loggedErrors.add(message)) {
            System.err.println("Error: " + message);
        }
    }
}