package tech.atani.client.util.game.render.animation;

import it.unimi.dsi.fastutil.doubles.Double2DoubleFunction;
import lombok.Builder;
import lombok.Getter;
import lombok.experimental.Accessors;

import java.time.Duration;

@Getter
@Accessors(fluent = true)
public class Animate {
	private float value;
	private final float minimum;
	private final float maximum;
	private final Duration speed;
	private EasingType easingType;
	private boolean reversed;
	private long startTime;

	@Builder
	private Animate(float minimum, float maximum, Duration speed, EasingType easingType, boolean reversed) {
		this.minimum = minimum;
		this.maximum = maximum;
		this.speed = speed != null ? speed : Duration.ofMillis(150);
		this.easingType = easingType != null ? easingType : EasingType.LINEAR;
		this.reversed = reversed;
		this.value = reversed ? maximum : minimum;
		this.startTime = System.nanoTime();
	}

	public Animate reversed(boolean reversed) {
		if (this.reversed != reversed) {
			this.reversed = reversed;
			this.value = reversed ? maximum : minimum;
			this.startTime = System.nanoTime();
		}
		return this;
	}

	public Animate easingType(EasingType easingType) {
		this.easingType = easingType;
		return this;
	}

	public void update() {
		long currentTime = System.nanoTime();
		long elapsedNanos = currentTime - startTime;
		long durationNanos = speed.toNanos();

		float progress = (float) elapsedNanos / durationNanos;
		progress = Math.max(0, Math.min(1, progress));
		progress = reversed ? 1 - progress : progress;

		float easedProgress = (float) easingType.function().get(progress);
		value = minimum + (maximum - minimum) * easedProgress;
	}

	public enum EasingType {
		LINEAR(x -> x),
		QUADRATIC_IN(x -> x * x),
		QUADRATIC_OUT(x -> x * (2 - x)),
		QUADRATIC_IN_OUT(x -> x < 0.5 ? 2 * x * x : -1 + (4 - 2 * x) * x),
		CUBIC_IN(x -> x * x * x),
		CUBIC_OUT(x -> 1 - Math.pow(1 - x, 3)),
		CUBIC_IN_OUT(x -> x < 0.5 ? 4 * x * x * x : 1 - Math.pow(-2 * x + 2, 3) / 2),
		QUARTIC_IN(x -> x * x * x * x),
		QUARTIC_OUT(x -> 1 - Math.pow(1 - x, 4)),
		QUARTIC_IN_OUT(x -> x < 0.5 ? 8 * x * x * x * x : 1 - Math.pow(-2 * x + 2, 4) / 2),
		QUINTIC_IN(x -> x * x * x * x * x),
		QUINTIC_OUT(x -> 1 - Math.pow(1 - x, 5)),
		QUINTIC_IN_OUT(x -> x < 0.5 ? 16 * x * x * x * x * x : 1 - Math.pow(-2 * x + 2, 5) / 2),
		SINE_IN(x -> 1 - Math.cos(x * Math.PI / 2)),
		SINE_OUT(x -> Math.sin(x * Math.PI / 2)),
		SINE_IN_OUT(x -> (1 - Math.cos(x * Math.PI)) / 2),
		EXPONENTIAL_IN(x -> x == 0 ? 0 : Math.pow(2, 10 * (x - 1))),
		EXPONENTIAL_OUT(x -> x == 1 ? 1 : 1 - Math.pow(2, -10 * x)),
		EXPONENTIAL_IN_OUT(x -> x == 0 ? 0 : x == 1 ? 1 : x < 0.5 ? Math.pow(2, 20 * x - 10) / 2 : (2 - Math.pow(2, -20 * x + 10)) / 2),
		CIRCULAR_IN(x -> 1 - Math.sqrt(1 - x * x)),
		CIRCULAR_OUT(x -> Math.sqrt(1 - (x - 1) * (x - 1))),
		CIRCULAR_IN_OUT(x -> x < 0.5 ? (1 - Math.sqrt(1 - 4 * x * x)) / 2 : (1 + Math.sqrt(1 - (2 * x - 2) * (2 * x - 2))) / 2),
		BACK_IN(x -> {
			double c1 = 1.70158;
			double c3 = c1 + 1;
			return c3 * x * x * x - c1 * x * x;
		}),
		BACK_OUT(x -> {
			double c1 = 1.70158;
			double c3 = c1 + 1;
			return 1 + c3 * Math.pow(x - 1, 3) + c1 * Math.pow(x - 1, 2);
		}),
		BACK_IN_OUT(x -> {
			double c1 = 1.70158;
			double c2 = c1 * 1.525;
			return x < 0.5 ? Math.pow(2 * x, 2) * ((c2 + 1) * 2 * x - c2) / 2 : (Math.pow(2 * x - 2, 2) * ((c2 + 1) * (x * 2 - 2) + c2) + 2) / 2;
		}),
		ELASTIC_IN(x -> {
			double c4 = 2 * Math.PI / 3;
			return x == 0 ? 0 : x == 1 ? 1 : -Math.pow(2, 10 * x - 10) * Math.sin((x * 10 - 10.75) * c4);
		}),
		ELASTIC_OUT(x -> {
			double c4 = 2 * Math.PI / 3;
			return x == 0 ? 0 : x == 1 ? 1 : Math.pow(2, -10 * x) * Math.sin((x * 10 - 0.75) * c4) + 1;
		}),
		ELASTIC_IN_OUT(x -> {
			double c5 = 2 * Math.PI / 4.5;
			double sin = Math.sin((20 * x - 11.125) * c5);
			return x == 0 ? 0 : x == 1 ? 1 : x < 0.5 ? -Math.pow(2, 20 * x - 10) * sin / 2 : Math.pow(2, -20 * x + 10) * sin / 2 + 1;
		}),
		BOUNCE_OUT(x -> {
			double n1 = 7.5625;
			double d1 = 2.75;
			if (x < 1 / d1) return n1 * x * x;
			else if (x < 2 / d1) return n1 * (x -= 1.5 / d1) * x + 0.75;
			else if (x < 2.5 / d1) return n1 * (x -= 2.25 / d1) * x + 0.9375;
			else return n1 * (x -= 2.625 / d1) * x + 0.984375;
		}),
		BOUNCE_IN(x -> 1 - BOUNCE_OUT.function.get(1 - x)),
		BOUNCE_IN_OUT(x -> x < 0.5 ? (1 - BOUNCE_OUT.function.get(1 - 2 * x)) / 2 : (1 + BOUNCE_OUT.function.get(2 * x - 1)) / 2);

		private final Double2DoubleFunction function;

		EasingType(Double2DoubleFunction function) {
			this.function = function;
		}

		public Double2DoubleFunction function() {
			return function;
		}
	}
}