package tech.atani.client.util.game.render;

import net.minecraft.client.gui.ScaledResolution;
import net.minecraft.client.renderer.GLAllocation;
import net.minecraft.client.renderer.culling.Frustum;
import net.minecraft.entity.Entity;
import net.minecraft.util.AxisAlignedBB;
import org.lwjgl.BufferUtils;
import org.lwjgl.util.glu.GLU;
import org.lwjgl.util.vector.Vector3f;
import org.lwjgl.util.vector.Vector4f;
import tech.atani.client.util.Util;
import tech.atani.client.util.system.math.MathUtil;

import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.util.Arrays;
import java.util.List;

import static org.lwjgl.opengl.GL45.*;

public class ProjectionUtil extends Util {
    private static final Frustum frustum = new Frustum();

    private static final FloatBuffer windowPosition = BufferUtils.createFloatBuffer(4);

    private static final IntBuffer viewport = GLAllocation.createDirectIntBuffer(16);
    private static final FloatBuffer modelMatrix = GLAllocation.createDirectFloatBuffer(16);
    private static final FloatBuffer projectionMatrix = GLAllocation.createDirectFloatBuffer(16);

    public static boolean isInView(Entity ent) {
        frustum.setPosition(mc.getRenderViewEntity().posX, mc.getRenderViewEntity().posY, mc.getRenderViewEntity().posZ);
        return frustum.isBoundingBoxInFrustum(ent.getEntityBoundingBox()) || ent.ignoreFrustumCheck;
    }

    public static Vector3f projectOn2D(float x, float y, float z, int scaleFactor) {
        glGetFloatv(GL_MODELVIEW_MATRIX, modelMatrix);
        glGetFloatv(GL_PROJECTION_MATRIX, projectionMatrix);

        glGetIntegerv(GL_VIEWPORT, viewport);

        if (GLU.gluProject(x, y, z, modelMatrix, projectionMatrix, viewport, windowPosition)) {
            return new Vector3f(windowPosition.get(0) / scaleFactor, (mc.displayHeight - windowPosition.get(1)) / scaleFactor, windowPosition.get(2));
        }

        return null;
    }

    public static double[] getInterpolatedPos(Entity entity) {
        float ticks = mc.timer.renderPartialTicks;

        return new double[]{
                MathUtil.interpolate(entity.lastTickPosX, entity.posX, ticks) - mc.getRenderManager().viewerPosX,
                MathUtil.interpolate(entity.lastTickPosY, entity.posY, ticks) - mc.getRenderManager().viewerPosY,
                MathUtil.interpolate(entity.lastTickPosZ, entity.posZ, ticks) - mc.getRenderManager().viewerPosZ
        };
    }

    public static AxisAlignedBB getInterpolatedBoundingBox(Entity entity) {
        final double[] renderingEntityPos = getInterpolatedPos(entity);

        final double entityRenderWidth = entity.width / 2;
        final double entityRenderHeight = entity.height;

        return new AxisAlignedBB(
                renderingEntityPos[0] - entityRenderWidth,
                renderingEntityPos[1],
                renderingEntityPos[2] - entityRenderWidth,

                renderingEntityPos[0] + entityRenderWidth,
                renderingEntityPos[1] + entityRenderHeight,
                renderingEntityPos[2] + entityRenderWidth
        ).expand(0.15, 0.15, 0.15);
    }

    public static Vector4f getEntityPositionsOn2D(Entity entity) {
        final AxisAlignedBB aabb = getInterpolatedBoundingBox(entity);

        final List<Vector3f> vectors = Arrays.asList(
                new Vector3f((float) aabb.minX, (float) aabb.minY, (float) aabb.minZ),
                new Vector3f((float) aabb.minX, (float) aabb.maxY, (float) aabb.minZ),
                new Vector3f((float) aabb.maxX, (float) aabb.minY, (float) aabb.minZ),
                new Vector3f((float) aabb.maxX, (float) aabb.maxY, (float) aabb.minZ),
                new Vector3f((float) aabb.minX, (float) aabb.minY, (float) aabb.maxZ),
                new Vector3f((float) aabb.minX, (float) aabb.maxY, (float) aabb.maxZ),
                new Vector3f((float) aabb.maxX, (float) aabb.minY, (float) aabb.maxZ),
                new Vector3f((float) aabb.maxX, (float) aabb.maxY, (float) aabb.maxZ)
        );

        Vector4f position = new Vector4f(Float.MAX_VALUE, Float.MAX_VALUE, -1.0f, -1.0f);
        ScaledResolution scaledResolution = new ScaledResolution(mc);

        for (Vector3f vector3f : vectors) {
            vector3f = projectOn2D(vector3f.x, vector3f.y, vector3f.z, scaledResolution.getScaleFactor());

            if (vector3f != null && vector3f.z >= 0.0 && vector3f.z < 1.0) {
                position.x = Math.min(vector3f.x, position.x);
                position.y = Math.min(vector3f.y, position.y);
                position.z = Math.max(vector3f.x, position.z);
                position.w = Math.max(vector3f.y, position.w);
            }
        }

        return position;
    }
}
