package tech.atani.client.storage.impl;

import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;
import lombok.Getter;
import tech.atani.client.storage.Storage;
import tech.atani.client.util.client.interfaces.ILogger;
import tech.atani.client.util.game.render.font.gui.font.ScaledFont;
import tech.atani.client.util.game.render.font.gui.font.glyph.SdfAtlas;
import tech.atani.client.util.game.render.font.resource.ResourceManager;
import tech.atani.client.util.game.render.font.resource.impl.ClassPathResourceManager;
import tech.atani.client.util.game.render.font.utils.StreamUtils;
import tech.atani.client.util.system.files.FileUtil;

import java.awt.*;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

@Getter
public class FontStorage extends Storage<String> {
    private static final String FRAGMENT_SHADER_PATH = "shaders/sdf/sdf.frag";
    private static final String VERTEX_SHADER_PATH = "shaders/sdf/textured_shape.vert";

    private static final String LINE_FRAGMENT_SHADER_PATH = "shaders/sdf/color_fill.frag";
    private static final String LINE_VERTEX_SHADER_PATH = "shaders/sdf/basic_shape.vert";

    private static final String FONT_DIRECTORY = "fonts/";

    private static final int FONT_BASE_SIZE = 512;
    private static final int SDF_ATLAS_DOWNSCALE = 8;
    private static final int SDF_ATLAS_SPREAD = 40;

    private static final char[] chars;

    static {
        List<Character> tempChars = new ArrayList<>();

        for (int i = 0x0020; i <= 0x007E; i++) tempChars.add((char) i); // Basic Latin
        for (int i = 0x00A0; i <= 0x00FF; i++) tempChars.add((char) i); // Latin-1 Supplement

        int[] latvianChars = {
                0x0100, 0x0101, // Ā ā
                0x010C, 0x010D, // Č č
                0x0112, 0x0113, // Ē ē
                0x0122, 0x0123, // Ģ ģ
                0x012A, 0x012B, // Ī ī
                0x0136, 0x0137, // Ķ ķ
                0x013B, 0x013C, // Ļ ļ
                0x0145, 0x0146, // Ņ ņ
                0x0160, 0x0161, // Š š
                0x0179, 0x017A, // Ź ź
                0x017B, 0x017C, // Ż ż
                0x017D, 0x017E, // Ž ž
                0x016A, 0x016B  // Ū ū
        };

        for (int code : latvianChars) {
            tempChars.add((char) code);
        }

        // Add Cyrillic block: U+0400 to U+04FF
        for (int i = 0x0400; i <= 0x04FF; i++) {
            tempChars.add((char) i);
        }

        tempChars.add('●');

        chars = new char[tempChars.size()];
        for (int i = 0; i < chars.length; i++) chars[i] = tempChars.get(i);
    }

    private final String sdfFragShaderSource;
    private final String sdfVertShaderSource;
    private final String lineFragShaderSource;
    private final String lineVertShaderSource;

    private final ResourceManager resourceManager;
    private final List<tech.atani.client.util.game.render.font.gui.font.Font> fonts = new ArrayList<>();
    private Path metadataFile;
    private Path atlasesDirectory;

    public FontStorage() {
        this.resourceManager = new ClassPathResourceManager("/assets/atani/");

        try {
            InputStream fragShaderStream = resourceManager.getInputStream(FRAGMENT_SHADER_PATH);
            InputStream vertShaderStream = resourceManager.getInputStream(VERTEX_SHADER_PATH);

            InputStream lineFragShaderStream = resourceManager.getInputStream(LINE_FRAGMENT_SHADER_PATH);
            InputStream lineVertShaderStream = resourceManager.getInputStream(LINE_VERTEX_SHADER_PATH);

            if (fragShaderStream == null) {
                throw new FileNotFoundException(FRAGMENT_SHADER_PATH);
            }
            if (vertShaderStream == null) {
                throw new FileNotFoundException(VERTEX_SHADER_PATH);
            }

            if (lineFragShaderStream == null) {
                throw new FileNotFoundException(LINE_FRAGMENT_SHADER_PATH);
            }
            if (lineVertShaderStream == null) {
                throw new FileNotFoundException(LINE_VERTEX_SHADER_PATH);
            }

            sdfFragShaderSource = StreamUtils.readStreamAsUtf8(fragShaderStream);
            sdfVertShaderSource = StreamUtils.readStreamAsUtf8(vertShaderStream);

            lineFragShaderSource = StreamUtils.readStreamAsUtf8(lineFragShaderStream);
            lineVertShaderSource = StreamUtils.readStreamAsUtf8(lineVertShaderStream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void init() {
        Path clientPath = FileUtil.getRunningPath();
        Path fontCacheDirectory = clientPath.resolve("cache");
        metadataFile = fontCacheDirectory.resolve("metadata.json");
        atlasesDirectory = fontCacheDirectory.resolve("atlases");

        try {
            Files.createDirectories(atlasesDirectory);
        } catch (IOException e) {
            ILogger.logger.warn("Failed to create font cache atlases directory: {}", e.getMessage());
            return;
        }

        try {
            loadFontCache();
        } catch (IOException e) {
            ILogger.logger.warn("Failed to load font cache metadata: {}", e.getMessage());
        }

        autoLoadFonts();
    }

    private void autoLoadFonts() {
        Set<String> loadedFonts = fonts.stream().map(tech.atani.client.util.game.render.font.gui.font.Font::getName).collect(Collectors.toSet());
        List<String> availableFonts = getAvailableFontFiles();

        for (String fontName : availableFonts) {
            if (!loadedFonts.contains(fontName)) {
                loadFont(fontName);
            }
        }

        saveFontCache();
    }

    private List<String> getAvailableFontFiles() {
        List<String> fontNames = new ArrayList<>();
        try {
            List<String> fontFiles = resourceManager.list(FONT_DIRECTORY);
            for (String fontFile : fontFiles) {
                if (fontFile.endsWith(".ttf") || fontFile.endsWith(".otf")) {
                    fontNames.add(fontFile.replace(".ttf", "").replace(".otf", ""));
                }
            }
        } catch (IOException e) {
            ILogger.logger.warn("Failed to list fonts in assets directory: {}", e.getMessage());
        }
        return fontNames;
    }

    private void loadFontCache() throws IOException {
        if (!Files.isRegularFile(metadataFile)) return;

        try (JsonReader reader = new JsonReader(Files.newBufferedReader(metadataFile))) {
            reader.beginArray();
            while (reader.hasNext()) {
                try {
                    fonts.add(tech.atani.client.util.game.render.font.gui.font.Font.read(this, reader, atlasesDirectory));
                } catch (IOException e) {
                    ILogger.logger.warn("Failed to load font from cache: {}", e.getMessage());
                }
            }
            reader.endArray();
        }
    }

    private void saveFontCache() {
        try (JsonWriter writer = new JsonWriter(Files.newBufferedWriter(metadataFile))) {
            writer.setIndent("  ");
            writer.beginArray();
            for (tech.atani.client.util.game.render.font.gui.font.Font font : fonts) {
                font.write(writer, atlasesDirectory);
            }
            writer.endArray();
        } catch (IOException e) {
            ILogger.logger.warn("Failed to save font cache: {}", e.getMessage());
        }
    }

    private tech.atani.client.util.game.render.font.gui.font.Font loadFont(String name) {
        java.awt.Font awtFont = getAwtFont(name);
        if (awtFont == null) {
            ILogger.logger.warn("Font '{}' could not be loaded", name);
            return null;
        }

        awtFont = awtFont.deriveFont(java.awt.Font.PLAIN, FONT_BASE_SIZE);
        SdfAtlas sdfAtlas = SdfAtlas.fromFont(awtFont, chars, SDF_ATLAS_SPREAD, SDF_ATLAS_DOWNSCALE);
        tech.atani.client.util.game.render.font.gui.font.Font font = new tech.atani.client.util.game.render.font.gui.font.Font(this, name, sdfAtlas);
        fonts.add(font);
        return font;
    }

    private java.awt.Font getAwtFont(String fontName) {
        String[] extensions = {".ttf", ".otf"};
        for (String ext : extensions) {
            try (InputStream is = resourceManager.getInputStream(FONT_DIRECTORY + fontName + ext)) {
                if (is != null) {
                    return java.awt.Font.createFont(java.awt.Font.TRUETYPE_FONT, is);
                }
            } catch (IOException | FontFormatException e) {
                ILogger.logger.warn("Failed to load font {}: {}", fontName + ext, e.getMessage());
            }
        }
        return null;
    }

    public ScaledFont getFont(String name, float size) {
        return fonts.stream()
                .filter(font -> font.getName().equalsIgnoreCase(name))
                .findFirst()
                .orElseGet(() -> {
                    tech.atani.client.util.game.render.font.gui.font.Font font = loadFont(name);
                    if (font != null) saveFontCache();
                    return font;
                }).newScaledFont(size, tech.atani.client.util.game.render.font.gui.font.Font.WEIGHT_REGULAR);
    }
}
