package tech.atani.client.setting;

import lombok.Getter;
import lombok.NonNull;
import tech.atani.client.Atani;
import tech.atani.client.setting.data.ISettingListener;
import tech.atani.client.util.client.language.TranslationUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.function.BooleanSupplier;

@Getter
public abstract class Setting<T> {
    private final List<ISettingListener<T>> listeners = new ArrayList<>();
    private final String name;
    private T value;
    private BooleanSupplier hidden = () -> false;

    public Setting(@NonNull String name, @NonNull T value) {
        this(name, value, Atani.instance.moduleStorage.newestModule.getSettings());
    }

    public Setting(@NonNull String name, @NonNull T value, @NonNull List<Setting<?>> collection) {
        this.name = name;
        this.value = value;
        collection.add(this);
    }

    public String getName() {
        return TranslationUtil.getTranslation(name);
    }

    public String getUntranslatedName() {
        return name;
    }

    public void setValue(@NonNull T value) {
        if (this.value != value) {
            T oldValue = this.value;
            this.value = value;
            listeners.forEach(listener -> listener.onChange(this, oldValue, value));
        }
    }

    public String stringValue() {
        return String.valueOf(value);
    }

    public int intValue() {
        throw new UnsupportedOperationException("Invalid setting value type: Integer");
    }

    public float floatValue() {
        throw new UnsupportedOperationException("Invalid setting value type: Float");
    }

    public double doubleValue() {
        throw new UnsupportedOperationException("Invalid setting value type: Double");
    }

    public boolean boolValue() {
        throw new UnsupportedOperationException("Invalid setting value type: Boolean");
    }

    public void addListener(ISettingListener<T> listener) {
        listeners.add(listener);
    }

    @SuppressWarnings("unchecked")
    public <I extends Setting<?>> I hide(BooleanSupplier hidden) {
        this.hidden = hidden != null ? hidden : () -> false;
        return (I) this;
    }
}