package tech.atani.client.module.scripting;

import net.minecraft.entity.EntityLivingBase;
import org.luaj.vm2.*;
import org.luaj.vm2.lib.OneArgFunction;
import org.luaj.vm2.lib.ThreeArgFunction;
import org.luaj.vm2.lib.VarArgFunction;
import org.luaj.vm2.lib.ZeroArgFunction;
import tech.atani.client.util.client.interfaces.IMinecraft;
import tech.atani.client.util.game.player.MovementUtil;

public class ScriptAPIFunctions implements IMinecraft {
    public static class Player extends LuaTable {
        private static final LuaTable META_TABLE = createMetaTable();

        public Player() {
            setmetatable(META_TABLE);
            registerFunctions();
        }

        private static LuaTable createMetaTable() {
            var metaTable = new LuaTable();
            metaTable.set("__index", new LuaFunction() {
                @Override
                public LuaValue call(LuaValue table, LuaValue key) {
                    if (mc.thePlayer == null) {
                        return switch (key.tojstring()) {
                            case "onGround", "moving" -> LuaValue.FALSE;
                            case "motionX", "motionY", "motionZ", "speed", "yaw", "pitch",
                                 "eyeHeight", "posX", "posY", "posZ" -> LuaValue.valueOf(0.0);
                            default -> LuaValue.NIL;
                        };
                    }
                    return switch (key.tojstring()) {
                        case "onGround" -> LuaValue.valueOf(mc.thePlayer.onGround);
                        case "moving" -> LuaValue.valueOf(MovementUtil.isMoving());
                        case "motionX" -> LuaValue.valueOf(mc.thePlayer.motionX);
                        case "motionY" -> LuaValue.valueOf(mc.thePlayer.motionY);
                        case "motionZ" -> LuaValue.valueOf(mc.thePlayer.motionZ);
                        case "speed" -> LuaValue.valueOf(MovementUtil.getSpeed());
                        case "yaw" -> LuaValue.valueOf(mc.thePlayer.rotationYaw);
                        case "pitch" -> LuaValue.valueOf(mc.thePlayer.rotationPitch);
                        case "eyeHeight" -> LuaValue.valueOf(mc.thePlayer.getEyeHeight());
                        case "posX" -> LuaValue.valueOf(mc.thePlayer.posX);
                        case "posY" -> LuaValue.valueOf(mc.thePlayer.posY);
                        case "posZ" -> LuaValue.valueOf(mc.thePlayer.posZ);
                        default -> LuaValue.NIL;
                    };
                }
            });
            return metaTable;
        }

        private void registerFunctions() {
            set("jump", new ZeroArgFunction() {
                @Override public LuaValue call() {
                    if (mc.thePlayer != null) mc.thePlayer.jump();
                    return LuaValue.NIL;
                }
            });
            set("stop", new ZeroArgFunction() {
                @Override public LuaValue call() {
                    if (mc.thePlayer != null) MovementUtil.stop();
                    return LuaValue.NIL;
                }
            });
            set("setSprinting", new OneArgFunction() {
                @Override public LuaValue call(LuaValue sprinting) {
                    if (mc.thePlayer != null) mc.thePlayer.setSprinting(sprinting.toboolean());
                    return LuaValue.NIL;
                }
            });
            set("setSpeed", new OneArgFunction() {
                @Override public LuaValue call(LuaValue moveSpeed) {
                    if (mc.thePlayer != null) MovementUtil.setSpeed(moveSpeed.todouble());
                    return LuaValue.NIL;
                }
            });
            set("setMotionX", new OneArgFunction() {
                @Override public LuaValue call(LuaValue motionX) {
                    if (mc.thePlayer != null) mc.thePlayer.motionX = motionX.todouble();
                    return LuaValue.NIL;
                }
            });
            set("setMotionY", new OneArgFunction() {
                @Override public LuaValue call(LuaValue motionY) {
                    if (mc.thePlayer != null) mc.thePlayer.motionY = motionY.todouble();
                    return LuaValue.NIL;
                }
            });
            set("setMotionZ", new OneArgFunction() {
                @Override public LuaValue call(LuaValue motionZ) {
                    if (mc.thePlayer != null) mc.thePlayer.motionZ = motionZ.todouble();
                    return LuaValue.NIL;
                }
            });
            set("setMotion", new ThreeArgFunction() {
                @Override public LuaValue call(LuaValue x, LuaValue y, LuaValue z) {
                    if (mc.thePlayer != null) {
                        mc.thePlayer.motionX = x.todouble();
                        mc.thePlayer.motionY = y.todouble();
                        mc.thePlayer.motionZ = z.todouble();
                    }
                    return LuaValue.NIL;
                }
            });
            set("strafe", new VarArgFunction() {
                @Override public Varargs invoke(Varargs args) {
                    if (mc.thePlayer != null) {
                        if (args.narg() == 0) MovementUtil.strafe();
                        else MovementUtil.strafe(args.arg1().todouble());
                    }
                    return LuaValue.NIL;
                }
            });
            set("sendChat", new OneArgFunction() {
                @Override public LuaValue call(LuaValue message) {
                    if (mc.thePlayer != null) IMinecraft.chat(message.tojstring());
                    return LuaValue.NIL;
                }
            });
            set("clickMouse", new ZeroArgFunction() {
                @Override public LuaValue call() {
                    mc.clickMouse();
                    return LuaValue.NIL;
                }
            });
            set("setYaw", new OneArgFunction() {
                @Override public LuaValue call(LuaValue yaw) {
                    if (mc.thePlayer != null) mc.thePlayer.rotationYaw = yaw.tofloat();
                    return LuaValue.NIL;
                }
            });
            set("setPitch", new OneArgFunction() {
                @Override public LuaValue call(LuaValue pitch) {
                    if (mc.thePlayer != null) mc.thePlayer.rotationPitch = pitch.tofloat();
                    return LuaValue.NIL;
                }
            });
        }
    }

    public static class Entity extends LuaTable {
        private static final LuaTable META_TABLE = createMetaTable();

        public Entity(net.minecraft.entity.Entity entity) {
            rawset("entity", LuaValue.userdataOf(entity));
            setmetatable(META_TABLE);
        }

        private static LuaTable createMetaTable() {
            var metaTable = new LuaTable();
            metaTable.set("__index", new LuaFunction() {
                @Override
                public LuaValue call(LuaValue table, LuaValue key) {
                    var entity = (net.minecraft.entity.Entity) table.rawget("entity").touserdata();
                    if (entity == null) return LuaValue.NIL;

                    return switch (key.tojstring()) {
                        case "name" -> LuaValue.valueOf(entity.getName());
                        case "health" -> entity instanceof EntityLivingBase living ?
                                LuaValue.valueOf(living.getHealth()) : LuaValue.NIL;
                        case "maxHealth" -> entity instanceof EntityLivingBase living ?
                                LuaValue.valueOf(living.getMaxHealth()) : LuaValue.NIL;
                        case "posX" -> LuaValue.valueOf(entity.posX);
                        case "posY" -> LuaValue.valueOf(entity.posY);
                        case "posZ" -> LuaValue.valueOf(entity.posZ);
                        case "yaw" -> LuaValue.valueOf(entity.rotationYaw);
                        case "pitch" -> LuaValue.valueOf(entity.rotationPitch);
                        case "eyeHeight" -> LuaValue.valueOf(entity.getEyeHeight());
                        case "distanceToPlayer" -> LuaValue.valueOf(mc.thePlayer != null ?
                                mc.thePlayer.getDistanceToEyes(entity) : 0.0);
                        default -> LuaValue.NIL;
                    };
                }
            });
            return metaTable;
        }
    }

    public static class World extends LuaTable {
        private static final LuaTable META_TABLE = createMetaTable();

        public World() {
            setmetatable(META_TABLE);
        }

        private static LuaTable createMetaTable() {
            var metaTable = new LuaTable();
            metaTable.set("__index", new LuaFunction() {
                @Override
                public LuaValue call(LuaValue table, LuaValue key) {
                    if (mc.theWorld == null || !"entities".equals(key.tojstring())) {
                        return LuaValue.tableOf();
                    }
                    var entitiesTable = new LuaTable();
                    var entityList = mc.theWorld.loadedEntityList;
                    int index = 1;
                    for (var entity : entityList) {
                        if (entity instanceof EntityLivingBase) {
                            entitiesTable.set(index++, new Entity(entity));
                        }
                    }
                    return entitiesTable;
                }
            });
            return metaTable;
        }
    }

    public static class Settings extends LuaTable {
        private static final LuaTable META_TABLE = createMetaTable();

        public Settings() {
            setmetatable(META_TABLE);
        }

        private static LuaTable createMetaTable() {
            var metaTable = new LuaTable();
            metaTable.set("__index", new LuaFunction() {
                @Override
                public LuaValue call(LuaValue table, LuaValue key) {
                    if (mc.thePlayer == null) {
                        return switch (key.tojstring()) {
                            case "jumpDown", "jumpPressed", "sneakDown", "sneakPressed",
                                 "forwardDown", "forwardPressed", "backDown", "backPressed",
                                 "rightDown", "rightPressed", "leftDown", "leftPressed" -> LuaValue.FALSE;
                            default -> LuaValue.NIL;
                        };
                    }
                    return switch (key.tojstring()) {
                        case "jumpDown" -> LuaValue.valueOf(mc.gameSettings.keyBindJump.isKeyDown());
                        case "jumpPressed" -> LuaValue.valueOf(mc.gameSettings.keyBindJump.isPressed());
                        case "sneakDown" -> LuaValue.valueOf(mc.gameSettings.keyBindSneak.isKeyDown());
                        case "sneakPressed" -> LuaValue.valueOf(mc.gameSettings.keyBindSneak.isPressed());
                        case "forwardDown" -> LuaValue.valueOf(mc.gameSettings.keyBindForward.isKeyDown());
                        case "forwardPressed" -> LuaValue.valueOf(mc.gameSettings.keyBindForward.isPressed());
                        case "backDown" -> LuaValue.valueOf(mc.gameSettings.keyBindBack.isKeyDown());
                        case "backPressed" -> LuaValue.valueOf(mc.gameSettings.keyBindBack.isPressed());
                        case "rightDown" -> LuaValue.valueOf(mc.gameSettings.keyBindRight.isKeyDown());
                        case "rightPressed" -> LuaValue.valueOf(mc.gameSettings.keyBindRight.isPressed());
                        case "leftDown" -> LuaValue.valueOf(mc.gameSettings.keyBindLeft.isKeyDown());
                        case "leftPressed" -> LuaValue.valueOf(mc.gameSettings.keyBindLeft.isPressed());
                        default -> LuaValue.NIL;
                    };
                }
            });
            return metaTable;
        }
    }

    public void load(Globals globals) {
        globals.set("player", new Player());
        globals.set("settings", new Settings());
        globals.set("world", new World());
    }
}