package tech.atani.client.module.scripting;

import org.luaj.vm2.LuaValue;
import org.luaj.vm2.lib.jse.JsePlatform;
import tech.atani.client.Atani;
import tech.atani.client.authentication.Connector;
import tech.atani.client.authentication.request.impl.ScriptsRequest;
import tech.atani.client.event.data.State;
import tech.atani.client.event.impl.UpdateEvent;
import tech.atani.client.module.Module;
import tech.atani.client.module.data.Category;
import tech.atani.client.module.data.ModuleData;
import tech.atani.client.util.client.events.base.Listen;
import tech.atani.client.util.client.interfaces.IClient;
import tech.atani.client.util.client.interfaces.ILogger;
import tech.atani.client.util.client.interfaces.ISubscriber;
import tech.atani.client.util.system.files.FileUtil;

import java.io.IOException;
import java.lang.reflect.Proxy;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class ScriptAPI implements IClient, ISubscriber, ILogger {
    private static final ExecutorService SCRIPT_EXECUTOR = Executors.newFixedThreadPool(2);

    public void loadScripts() {
        Path scriptPath = Path.of(FileUtil.getRunningPath().toString(), "scripts");
        if (!Files.exists(scriptPath)) {
            try {
                Files.createDirectories(scriptPath);
            } catch (IOException e) {
                logger.error("Failed to create scripts folder: {}", e.getMessage());
            }
            return;
        }

        SCRIPT_EXECUTOR.submit(() -> {
            Connector.instance.requestStorage.getT(ScriptsRequest.class).request("list");
            try (var stream = Files.list(scriptPath)) {
                stream.filter(p -> p.toString().endsWith(".lua")).forEach(this::processScript);
            } catch (IOException e) {
                logger.error("Failed to list scripts: {}", e.getMessage());
            }
        });
    }

    private void processScript(Path scriptPath) {
        String content;
        try {
            content = Files.readString(scriptPath);
        } catch (IOException e) {
            logger.warn("Failed to read script {}: {}", scriptPath.getFileName(), e.getMessage());
            return;
        }

        if (!ScriptAPIValidator.isScriptSafe(content)) {
            return;
        }

        var globals = JsePlatform.standardGlobals();
        new ScriptAPIFunctions().load(globals);

        LuaValue chunk = globals.load(content, scriptPath.getFileName().toString());
        chunk.call();

        var name = globals.get("name");
        var desc = globals.get("description");

        if (name.isnil() || desc.isnil()) {
            logger.warn("Skipping {}: missing name or description", scriptPath.getFileName());
            return;
        }

        var scriptData = createModuleData(name.tojstring(), desc.tojstring());
        var scriptModule = new Module(scriptData) {
            @Override
            public void onEnable() {
                var luaOnEnable = globals.get("onEnable");
                if (!luaOnEnable.isnil()) luaOnEnable.call();
            }

            @Override
            public void onDisable() {
                var luaOnDisable = globals.get("onDisable");
                if (!luaOnDisable.isnil()) luaOnDisable.call();
            }

            @Listen
            @SuppressWarnings("unused")
            private void onUpdate(UpdateEvent event) {
                if (event.state == State.PRE) {
                    var luaOnUpdate = globals.get("onUpdate");
                    if (!luaOnUpdate.isnil()) luaOnUpdate.call();
                }
            }
        };
        Atani.instance.moduleStorage.add(scriptModule);
    }

    private ModuleData createModuleData(String name, String desc) {
        Map<String, Object> values = new ConcurrentHashMap<>(Map.of(
                "name", name,
                "category", Category.SCRIPT,
                "description", desc
        ));

        return (ModuleData) Proxy.newProxyInstance(
                ModuleData.class.getClassLoader(),
                new Class[]{ModuleData.class},
                (proxy, method, args) -> values.get(method.getName())
        );
    }
}