package tech.atani.client.module.impl.hud;

import lombok.RequiredArgsConstructor;
import tech.atani.client.Atani;
import tech.atani.client.event.impl.Render2DEvent;
import tech.atani.client.module.Module;
import tech.atani.client.module.data.Category;
import tech.atani.client.module.data.ModuleData;
import tech.atani.client.setting.impl.CheckBoxSetting;
import tech.atani.client.setting.impl.ColorSetting;
import tech.atani.client.setting.impl.SliderSetting;
import tech.atani.client.setting.impl.StringSetting;
import tech.atani.client.util.client.events.base.Listen;
import tech.atani.client.util.game.render.animation.Animate;
import tech.atani.client.util.game.render.font.gui.font.renderer.SimpleFontRenderer;
import tech.atani.client.util.game.render.shader.data.renderer.Renderer;

import java.awt.*;
import java.time.Duration;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@ModuleData(name = "ModuleList", category = Category.HUD, description = "modules.arraylist.description")
@SuppressWarnings("unused")
public class ModuleList extends Module {

    private final StringSetting fontFamily = new StringSetting.Builder()
            .name("modules.arraylist.font")
            .value("Mono")
            .values("Google", "Inter", "Mono")
            .build();

    private final StringSetting fontType = new StringSetting.Builder()
            .name("modules.arraylist.fonttype")
            .value("Regular")
            .values("Bold", "Medium", "Regular")
            .build();

    private final SliderSetting fontSize = new SliderSetting.Builder()
            .name("modules.arraylist.fontsize")
            .value(11)
            .min(8)
            .max(20)
            .build();

    private final CheckBoxSetting fontShadow = new CheckBoxSetting.Builder()
            .name("modules.arraylist.fontshadow")
            .value(true)
            .build();

    private final ColorSetting fontColor = new ColorSetting.Builder()
            .name("modules.arraylist.fontcolor")
            .value(new Color(255, 255, 255))
            .build();

    private final CheckBoxSetting background = new CheckBoxSetting.Builder()
            .name("modules.arraylist.background")
            .value(true)
            .build();

    private final SliderSetting yOffset = new SliderSetting.Builder()
            .name("modules.arraylist.yoffset")
            .value(2.5)
            .min(0)
            .max(10)
            .increment(1)
            .build();

    private final SliderSetting xOffset = new SliderSetting.Builder()
            .name("modules.arraylist.xoffset")
            .value(2.5)
            .min(0)
            .max(10)
            .increment(1)
            .build();

    private final CheckBoxSetting suffix = new CheckBoxSetting.Builder()
            .name("modules.arraylist.suffix")
            .value(false)
            .build();

    private final CheckBoxSetting spaceOut = new CheckBoxSetting.Builder()
            .name("modules.arraylist.spaceout")
            .value(false)
            .build();

    private final Map<Module, FeatureHolder> featureHolderMap = new HashMap<>();
    private SimpleFontRenderer font = Atani.instance.fontStorage.getFont("Mono-Regular", 11).newRenderer();

    public ModuleList() {
        fontFamily.addListener(((setting, oldValue, newValue) -> setFont()));
        fontType.addListener(((setting, oldValue, newValue) -> setFont()));
        fontSize.addListener(((setting, oldValue, newValue) -> setFont()));
    }

    private void setFont() {
        switch (fontFamily.stringValue()) {
            case "Inter" ->
                    font = Atani.instance.fontStorage.getFont("Inter-" + fontType.stringValue(), fontSize.intValue()).newRenderer();
            case "Google" ->
                    font = Atani.instance.fontStorage.getFont("Google-" + fontType.stringValue(), fontSize.intValue()).newRenderer();
            case "Mono" ->
                    font = Atani.instance.fontStorage.getFont("Mono-" + fontType.stringValue(), fontSize.intValue()).newRenderer();
        }
    }

    @Listen
    private void onEvent(Render2DEvent event) {
        List<FeatureHolder> sortedFeatures = Atani.instance.moduleStorage.stream()
                .map(module -> {
                    FeatureHolder holder = featureHolderMap.computeIfAbsent(module, FeatureHolder::new);
                    holder.updateAnimation();
                    return holder;
                })
                .filter(FeatureHolder::shouldShow)
                .sorted(Comparator.comparingDouble(holder -> font.getStringWidthPure(getModuleName(holder.feature, false))))
                .collect(Collectors.collectingAndThen(Collectors.toList(), list -> {
                    Collections.reverse(list);
                    return list;
                }));

        float y = yOffset.intValue() + 4;
        if (mc.gameSettings.showDebugInfo) {
            y = 137 + 2;
        }

        int count = 0;
        for (FeatureHolder holder : sortedFeatures) {
            float animValue = holder.getAnimationValue();
            String moduleName = getModuleName(holder.feature, false);

            float moduleWidth = font.getStringWidthPure(moduleName) + 4;
            float moduleWidth2 = font.getStringWidthPure(getModuleName(holder.feature, true)) + 4;
            float moduleHeight = font.getFontHeight();

            float x = event.getScaledResolution().getScaledWidth() - (moduleWidth + 4) * animValue - xOffset.getValue().intValue();
            float rectHeight = moduleHeight * animValue;
            float textY = y + (rectHeight - moduleHeight) / 2;

            float baseHeight = font.getFontHeight() - 4;
            float dynamicSpacing = Math.max(baseHeight * 1.3f, baseHeight - (fontSize.floatValue() - 14) * 0.5f) + (background.getValue() ? 2 : 0);

            if (background.getValue()) {
                drawBackground(x, y, moduleWidth, dynamicSpacing, count, sortedFeatures);
            }

            Color textColor2 = fontColor.getValue();
            Color textColor3 = fontColor.getValue().darker().darker();

            long time = System.currentTimeMillis();

            font.drawYCenteredGradientString(getModuleName(holder.feature, true), x + 2, y + dynamicSpacing / 2, time, textColor2, textColor3, fontShadow.getValue());
            if (suffix.getValue() && holder.feature.getSuffix() != null)
                font.drawYCenteredGradientString(holder.feature.getSuffix(), x + moduleWidth2, y + dynamicSpacing / 2, time, Color.WHITE, Color.GRAY, fontShadow.getValue());

            y += dynamicSpacing;
            count++;
        }
    }

    private void drawBackground(float x, float y, float width, float height, int count, List<FeatureHolder> sortedFeatures) {
        Renderer.drawRectangle(x, y, width, height, new Color(0, 0, 0, 100));
    }

    private String getModuleName(Module module, boolean onlyName) {
        String newName = spaceOut.getValue() ? module.getData().name().replaceAll("(?<!^)(?<![A-Z])([A-Z])", " $1") : module.getData().name();

        if (onlyName) {
            return newName;
        }

        if (module.getSuffix() != null && suffix.getValue()) {
            newName += " " + module.getSuffix();
        }

        return newName;
    }

    @RequiredArgsConstructor
    static class FeatureHolder {
        private final Module feature;
        private final Animate toggleAnimation =
                Animate.builder()
                        .easingType(Animate.EasingType.LINEAR)
                        .minimum(0.0f)
                        .maximum(1.0f)
                        .speed(Duration.ofMillis(150))
                        .build();

        public void updateAnimation() {
            toggleAnimation
                    .reversed(!feature.isEnabled())
                    .easingType(feature.isEnabled() ? Animate.EasingType.CUBIC_OUT : Animate.EasingType.CUBIC_IN)
                    .update();
        }

        public boolean shouldShow() {
            return toggleAnimation.value() > 0.0f;
        }

        public float getAnimationValue() {
            return toggleAnimation.value();
        }
    }
}