package tech.atani.client.module;

import lombok.Getter;
import lombok.NonNull;
import lombok.Setter;
import tech.atani.client.Atani;
import tech.atani.client.module.data.ModuleData;
import tech.atani.client.setting.Setting;
import tech.atani.client.util.Util;
import tech.atani.client.util.client.language.TranslationUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;

@Getter
@Setter
public abstract class Module extends Util {
    protected final List<Setting<?>> settings = new ArrayList<>();
    protected final ModuleData data;
    private int key;
    private final AtomicBoolean enabled = new AtomicBoolean(false);
    private boolean expanded;
    private boolean frozen;

    private Consumer<Boolean> enableHandler = (state) -> {};

    public Module() {
        ModuleData annotation = getClass().getAnnotation(ModuleData.class);
        if (annotation == null) {
            throw new RuntimeException("Module manifest not found!");
        }
        this.data = annotation;
        Atani.instance.moduleStorage.newestModule = this;
    }

    public Module(@NonNull ModuleData moduleData) {
        this.data = moduleData;
        Atani.instance.moduleStorage.newestModule = this;
    }

    public String getName() {
        return data.name();
    }

    public String getDesc() {
        return TranslationUtil.getTranslation(data.description());
    }

    public boolean isExpanded() {
        return !settings.isEmpty() && expanded;
    }

    public String getSuffix() {
        return null;
    }

    public void onEnable() {}
    public void onDisable() {}
    public void onInitialize() {}

    public void setEnabled(boolean state) {
        if (frozen) {
            return;
        }

        if (enabled.compareAndSet(!state, state)) {
            if (state) {
                this.onEnable();
                bus.register(this);
            } else {
                this.onDisable();
                bus.unregister(this);
            }
            enableHandler.accept(state);
        }
    }

    public void toggle() {
        if (frozen) {
            return;
        }

        setEnabled(!enabled.get());
    }

    public boolean isEnabled() {
        return enabled.get();
    }

    public boolean isDisabled() {
        return !enabled.get();
    }

    public void expand() {
        if (!settings.isEmpty()) {
            expanded = !expanded;
        }
    }

    public Setting<?> getSettingByName(String input) {
        return settings.stream()
                .filter(s -> s.getUntranslatedName().equalsIgnoreCase(input))
                .findFirst()
                .orElse(null);
    }
}
