package tech.atani.client.file.impl;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import tech.atani.client.Atani;
import tech.atani.client.file.File;
import tech.atani.client.module.Module;
import tech.atani.client.setting.Setting;
import tech.atani.client.setting.impl.*;
import tech.atani.client.util.system.files.FileUtil;

import java.util.Arrays;

public class ModuleFile extends File {
    private final Gson gson = new GsonBuilder().setPrettyPrinting().create();

    public ModuleFile(String name) {
        super(name, FileUtil.getRunningPath() + "/configs/" + name + ".ata");
    }

    @Override
    protected String getData() {
        JsonObject configObject = new JsonObject();
        for (Module module : Atani.instance.moduleStorage.getSet()) {
            JsonObject moduleObject = new JsonObject();

            moduleObject.addProperty("Enabled", module.isEnabled());

            JsonObject settingObject = new JsonObject();
            for (Setting<?> setting : module.getSettings()) {
                switch (setting) {
                    case SliderSetting s -> settingObject.addProperty(setting.getUntranslatedName(), s.getValue());
                    case StringSetting s -> settingObject.addProperty(setting.getUntranslatedName(), s.getValue());
                    case RangeSetting s ->
                            settingObject.addProperty(setting.getUntranslatedName(), s.getValue() + "-" + s.getSecondValue());
                    case CheckBoxSetting cb -> settingObject.addProperty(setting.getUntranslatedName(), cb.getValue());
                    case MultiSelectSetting ms ->
                            settingObject.addProperty(setting.getUntranslatedName(), Arrays.toString(ms.getValue()));
                    default -> settingObject.addProperty(setting.getUntranslatedName(), setting.stringValue());
                }
            }

            moduleObject.add("Settings", settingObject);

            configObject.add(module.getData().name(), moduleObject);
        }
        return gson.toJson(configObject);
    }

    @Override
    protected void loadData(String data) {
        if (data.isBlank()) {
            data = "{}";
        }

        JsonObject configObject = JsonParser.parseString(data).getAsJsonObject();
        for (String moduleName : configObject.keySet()) {
            JsonObject moduleObject = configObject.getAsJsonObject(moduleName);
            Module module = Atani.instance.moduleStorage.getModule(moduleName);

            if (module != null) {
                module.setEnabled(false);

                JsonObject settingObject = moduleObject.getAsJsonObject("Settings");
                for (String settingName : settingObject.keySet()) {
                    String value = settingObject.get(settingName).getAsString();
                    Setting<?> setting = module.getSettingByName(settingName);

                    if (setting != null) {
                        switch (setting) {
                            case SliderSetting s -> s.setValue(Double.parseDouble(value));
                            case StringSetting s -> s.setValue(value);
                            case RangeSetting s -> {
                                String[] parts = value.split("-");
                                s.setValue(Double.parseDouble(parts[0]));
                                s.setSecondValue(Double.parseDouble(parts[1]));
                            }
                            case CheckBoxSetting cb -> cb.setValue(Boolean.parseBoolean(value));
                            case MultiSelectSetting ms ->
                                    ms.setValue(value.replace("[", "").replace("]", "").split(", "));
                            default -> {
                            }
                        }
                    }
                }

                module.setEnabled(moduleObject.get("Enabled").getAsBoolean());
            }
        }
    }
}