package org.lwjgl.openal;

import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.lwjgl.openal.AL10.*;
import static org.lwjgl.openal.AL11.*;
import static org.lwjgl.openal.ALC10.*;
import static org.lwjgl.openal.ALC11.ALC_MONO_SOURCES;
import static org.lwjgl.openal.ALC11.ALC_STEREO_SOURCES;
import static org.lwjgl.openal.EXTEfx.*;
import static org.lwjgl.openal.SOFTHRTF.ALC_HRTF_SOFT;

public class SoundSystemOpenAL {
    public static long audioContext;
    public static long audioDevice;
    public final Map<String, Integer> soundSources = new HashMap<>();
    private final List<Integer> freeSources = new ArrayList<>();
    private static int reverbAuxSlot;
    private static final int ALC_RESAMPLER_SOFT = 0x1000;
    private static final int ALC_RESAMPLER_HIGHEST_QUALITY_SOFT = 3;
    private static final int MAX_SOURCES = 255;

    public static void create() {
        String defaultDeviceName = alcGetString(0, ALC_DEFAULT_DEVICE_SPECIFIER);
        audioDevice = alcOpenDevice(defaultDeviceName);
        if (audioDevice == 0L) throw new RuntimeException("Failed to open OpenAL device");
        int[] attrib = {ALC_HRTF_SOFT, ALC_TRUE, ALC_RESAMPLER_SOFT, ALC_RESAMPLER_HIGHEST_QUALITY_SOFT, ALC_FREQUENCY, 44100, ALC_MONO_SOURCES, 255, ALC_STEREO_SOURCES, 32, 0};
        audioContext = alcCreateContext(audioDevice, attrib);
        if (audioContext == 0L) throw new RuntimeException("Failed to create OpenAL context");
        alcMakeContextCurrent(audioContext);
        ALCCapabilities alcCapabilities = ALC.createCapabilities(audioDevice);
        ALCapabilities alCapabilities = AL.createCapabilities(alcCapabilities);
        if (!alCapabilities.OpenAL10) throw new RuntimeException("OpenAL 1.0 not supported");
        if (alcCapabilities.ALC_EXT_EFX) {
            int reverbEffect = alGenEffects();
            alEffecti(reverbEffect, AL_EFFECT_TYPE, AL_EFFECT_REVERB);
            alEffectf(reverbEffect, AL_REVERB_DECAY_TIME, 3.0f);
            alEffectf(reverbEffect, AL_REVERB_DENSITY, 0.8f);
            reverbAuxSlot = alGenAuxiliaryEffectSlots();
            alAuxiliaryEffectSloti(reverbAuxSlot, AL_EFFECTSLOT_EFFECT, reverbEffect);
        }
        alListenerf(AL_GAIN, 1.0f);
    }

    public void newSource(boolean priority, String name, URL url, String identifier, boolean looping, float x, float y, float z, int attenuation, float volume, boolean applyReverb) {
        Integer sourceId;
        if (!freeSources.isEmpty()) {
            sourceId = freeSources.remove(0);
            alSourceStop(sourceId);
            alSourcei(sourceId, AL_BUFFER, 0);
            alSourcef(sourceId, AL_PITCH, 1.0f);
            alSourcef(sourceId, AL_GAIN, 1.0f);
            alSource3f(sourceId, AL_POSITION, 0.0f, 0.0f, 0.0f);
            alSourcei(sourceId, AL_LOOPING, AL_FALSE);
        } else if (soundSources.size() + freeSources.size() < MAX_SOURCES) {
            sourceId = alGenSources();
            if (!alIsSource(sourceId)) return;
        } else {
            return;
        }
        soundSources.put(name, sourceId);
        alSource3f(sourceId, AL_POSITION, x, y, z);
        alSourcef(sourceId, AL_GAIN, Math.max(0.0f, Math.min(volume, 1.0f)));
        alSourcei(sourceId, AL_LOOPING, looping ? AL_TRUE : AL_FALSE);
        alSourcef(sourceId, AL_PITCH, 1.0f);
        switch (attenuation) {
            case 1:
                alSourcei(sourceId, AL_DISTANCE_MODEL, AL_LINEAR_DISTANCE);
                alSourcef(sourceId, AL_MAX_DISTANCE, 100.0f);
                alSourcef(sourceId, AL_ROLLOFF_FACTOR, 1.0f);
                alSourcef(sourceId, AL_REFERENCE_DISTANCE, 5.0f);
                break;
            case 2:
                alSourcei(sourceId, AL_DISTANCE_MODEL, AL_EXPONENT_DISTANCE);
                alSourcef(sourceId, AL_ROLLOFF_FACTOR, 1.0f);
                alSourcef(sourceId, AL_REFERENCE_DISTANCE, 5.0f);
                break;
            default:
                alSourcei(sourceId, AL_DISTANCE_MODEL, AL_NONE);
                break;
        }
        if (applyReverb && reverbAuxSlot != 0) alSource3i(sourceId, AL_AUXILIARY_SEND_FILTER, reverbAuxSlot, 0, AL_FILTER_NULL);
        checkOpenALError("setting up source properties");
    }

    public void newStreamingSource(boolean priority, String name, URL url, String identifier, boolean looping, float x, float y, float z, int attenuation, float volume, boolean applyReverb) {
        newSource(priority, name, url, identifier, looping, x, y, z, attenuation, volume, applyReverb);
    }

    public boolean playing(String name) {
        Integer sourceId = soundSources.get(name);
        if (sourceId == null) return false;
        int state = alGetSourcei(sourceId, AL_SOURCE_STATE);
        return state == AL_PLAYING;
    }

    public void cleanup() {
        for (int source : soundSources.values()) {
            if (alGetSourcei(source, AL_SOURCE_STATE) == AL_PLAYING) alSourceStop(source);
            alDeleteSources(source);
        }
        for (int source : freeSources) alDeleteSources(source);
        freeSources.clear();
        soundSources.clear();
        if (reverbAuxSlot != 0) {
            alDeleteAuxiliaryEffectSlots(reverbAuxSlot);
            reverbAuxSlot = 0;
        }
        if (audioContext != 0L) {
            alcMakeContextCurrent(0);
            alcDestroyContext(audioContext);
            audioContext = 0L;
        }
        if (audioDevice != 0L) {
            alcCloseDevice(audioDevice);
            audioDevice = 0L;
        }
    }

    public void play(String name) {
        Integer sourceId = soundSources.get(name);
        if (sourceId != null) {
            int state = alGetSourcei(sourceId, AL_SOURCE_STATE);
            int buffer = alGetSourcei(sourceId, AL_BUFFER);
            if (buffer != 0 && state != AL_PLAYING) {
                alSourcePlay(sourceId);
                checkOpenALError("alSourcePlay");
            }
        }
    }

    public void pause(String name) {
        Integer sourceId = soundSources.get(name);
        if (sourceId != null && alGetSourcei(sourceId, AL_SOURCE_STATE) == AL_PLAYING) alSourcePause(sourceId);
    }

    public void stop(String name) {
        Integer sourceId = soundSources.get(name);
        if (sourceId != null) alSourceStop(sourceId);
    }

    public void removeSource(String name) {
        Integer sourceId = soundSources.remove(name);
        if (sourceId != null) {
            alSourceStop(sourceId);
            alSourcei(sourceId, AL_BUFFER, 0);
            freeSources.add(sourceId);
        }
    }

    public void setPitch(String name, float pitch) {
        Integer sourceId = soundSources.get(name);
        if (sourceId != null) alSourcef(sourceId, AL_PITCH, Math.max(0.5f, Math.min(pitch, 2.0f)));
    }

    public void setVolume(String name, float volume) {
        Integer sourceId = soundSources.get(name);
        if (sourceId != null) alSourcef(sourceId, AL_GAIN, Math.max(0.0f, Math.min(volume, 1.0f)));
    }

    public void setMasterVolume(float volume) {
        alListenerf(AL_GAIN, Math.max(0.0f, Math.min(volume, 1.0f)));
    }

    public void setPosition(String name, float x, float y, float z) {
        Integer sourceId = soundSources.get(name);
        if (sourceId != null) alSource3f(sourceId, AL_POSITION, x, y, z);
    }

    public void setListenerPosition(float x, float y, float z) {
        alListener3f(AL_POSITION, x, y, z);
    }

    public void setListenerOrientation(float atX, float atY, float atZ, float upX, float upY, float upZ) {
        float[] orientation = {atX, atY, atZ, upX, upY, upZ};
        alListenerfv(AL_ORIENTATION, orientation);
    }

    public float getMasterVolume() {
        float[] volume = new float[1];
        alGetListenerf(AL_GAIN, volume);
        return volume[0];
    }

    private static void checkOpenALError(String operation) {
        int error = alGetError();
        if (error != AL_NO_ERROR) throw new RuntimeException("OpenAL error during " + operation + ": " + error);
    }
}