package net.minecraft.client.renderer;

import com.google.common.primitives.Floats;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.nio.ShortBuffer;
import java.util.Arrays;
import java.util.BitSet;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.client.renderer.vertex.VertexFormat;
import net.minecraft.client.renderer.vertex.VertexFormatElement;
import net.minecraft.src.Config;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumWorldBlockLayer;
import net.minecraft.util.MathHelper;
import net.optifine.SmartAnimations;
import net.optifine.render.RenderEnv;
import net.optifine.shaders.SVertexBuilder;
import net.optifine.util.TextureUtils;
import org.lwjgl.opengl.GL11;

public final class WorldRenderer {
    private ByteBuffer byteBuffer;
    public IntBuffer rawIntBuffer;
    private ShortBuffer rawShortBuffer;
    public FloatBuffer rawFloatBuffer;
    public int vertexCount;
    private VertexFormatElement vertexFormatElement;
    private int vertexFormatIndex;
    private boolean noColor;
    public int drawMode;
    private double xOffset;
    private double yOffset;
    private double zOffset;
    private VertexFormat vertexFormat;
    private boolean isDrawing;
    private EnumWorldBlockLayer blockLayer;
    private boolean[] drawnIcons = new boolean[256];
    private TextureAtlasSprite[] quadSprites;
    private TextureAtlasSprite[] quadSpritesPrev;
    private TextureAtlasSprite quadSprite;
    public SVertexBuilder sVertexBuilder;
    private RenderEnv renderEnv;
    public BitSet animatedSprites;
    private final BitSet animatedSpritesCached = new BitSet();
    private boolean modeTriangles;
    private ByteBuffer byteBufferTriangles;

    public WorldRenderer(int bufferSizeIn) {
        this.byteBuffer = GLAllocation.createDirectByteBuffer(bufferSizeIn << 2);
        this.rawIntBuffer = byteBuffer.asIntBuffer();
        this.rawShortBuffer = byteBuffer.asShortBuffer();
        this.rawFloatBuffer = byteBuffer.asFloatBuffer();
        this.sVertexBuilder = new SVertexBuilder();
        SVertexBuilder.initVertexBuilder(this);
    }

    public void growBuffer(int size) {
        if (size > rawIntBuffer.remaining()) {
            int oldCapacity = byteBuffer.capacity();
            int newCapacity = Math.max(oldCapacity + (size << 2), oldCapacity << 1);
            ByteBuffer newBuffer = GLAllocation.createDirectByteBuffer(newCapacity);
            byteBuffer.flip();
            newBuffer.put(byteBuffer);
            byteBuffer = newBuffer;
            rawFloatBuffer = byteBuffer.asFloatBuffer();
            rawIntBuffer = byteBuffer.asIntBuffer();
            rawShortBuffer = byteBuffer.asShortBuffer();

            if (quadSprites != null) {
                TextureAtlasSprite[] oldSprites = quadSprites;
                quadSprites = new TextureAtlasSprite[getBufferQuadSize()];
                System.arraycopy(oldSprites, 0, quadSprites, 0, Math.min(oldSprites.length, quadSprites.length));
                quadSpritesPrev = null;
            }
        }
    }

    public void sortVertexData(float x, float y, float z) {
        if (vertexFormat == null) return; // Prevent NPE if not initialized
        int quadCount = vertexCount >> 2;
        float[] distances = new float[quadCount];

        for (int i = 0; i < quadCount; i++) {
            distances[i] = getDistanceSq(rawFloatBuffer, x + (float)xOffset,
                    y + (float)yOffset, z + (float)zOffset,
                    vertexFormat.getIntegerSize(), i * vertexFormat.getNextOffset());
        }

        Integer[] indices = new Integer[quadCount];
        Arrays.setAll(indices, Integer::valueOf);
        Arrays.sort(indices, (a, b) -> Floats.compare(distances[b], distances[a]));

        int stride = vertexFormat.getNextOffset();
        int[] temp = new int[stride];
        BitSet bitset = new BitSet();

        for (int i = 0; i < indices.length; i = bitset.nextClearBit(i)) {
            int src = indices[i];
            if (src != i) {
                rawIntBuffer.position(src * stride);
                rawIntBuffer.get(temp);
                int dest = src;
                for (int next = indices[dest]; dest != i; next = indices[next]) {
                    IntBuffer slice = rawIntBuffer.slice(next * stride, stride);
                    rawIntBuffer.position(dest * stride);
                    rawIntBuffer.put(slice);
                    bitset.set(dest);
                    dest = next;
                }
                rawIntBuffer.position(i * stride);
                rawIntBuffer.put(temp);
            }
            bitset.set(i);
        }

        rawIntBuffer.rewind();
        if (quadSprites != null) {
            TextureAtlasSprite[] newSprites = new TextureAtlasSprite[quadCount];
            for (int i = 0; i < quadCount; i++) {
                newSprites[i] = quadSprites[indices[i]];
            }
            quadSprites = newSprites;
        }
    }

    public record State(int[] rawBuffer, VertexFormat vertexFormat, TextureAtlasSprite[] quadSprites) {
        public int getVertexCount() {
            return rawBuffer.length / vertexFormat.getIntegerSize();
        }
    }

    public State getVertexState() {
        int size = getBufferSize();
        int[] buffer = new int[size];
        rawIntBuffer.rewind();
        rawIntBuffer.get(buffer);
        return quadSprites == null
                ? new State(buffer, new VertexFormat(vertexFormat), null)
                : new State(buffer, new VertexFormat(vertexFormat), Arrays.copyOf(quadSprites, vertexCount >> 2));
    }

    public int getBufferSize() {
        return vertexCount * (vertexFormat != null ? vertexFormat.getIntegerSize() : 0);
    }

    private static float getDistanceSq(FloatBuffer buffer, float x, float y, float z, int stride, int offset) {
        float xAvg = (buffer.get(offset) + buffer.get(offset + stride) +
                buffer.get(offset + stride * 2) + buffer.get(offset + stride * 3)) * 0.25f - x;
        float yAvg = (buffer.get(offset + 1) + buffer.get(offset + stride + 1) +
                buffer.get(offset + stride * 2 + 1) + buffer.get(offset + stride * 3 + 1)) * 0.25f - y;
        float zAvg = (buffer.get(offset + 2) + buffer.get(offset + stride + 2) +
                buffer.get(offset + stride * 2 + 2) + buffer.get(offset + stride * 3 + 2)) * 0.25f - z;
        return xAvg * xAvg + yAvg * yAvg + zAvg * zAvg;
    }

    public void setVertexState(State state) {
        if (state == null) return;
        growBuffer(state.rawBuffer.length);
        rawIntBuffer.rewind();
        rawIntBuffer.put(state.rawBuffer);
        vertexCount = state.getVertexCount();
        vertexFormat = new VertexFormat(state.vertexFormat);

        if (state.quadSprites != null) {
            quadSprites = quadSpritesPrev != null ? quadSpritesPrev : new TextureAtlasSprite[getBufferQuadSize()];
            System.arraycopy(state.quadSprites, 0, quadSprites, 0, state.quadSprites.length);
        } else {
            quadSpritesPrev = quadSprites;
            quadSprites = null;
        }
    }

    public void reset() {
        vertexCount = 0;
        vertexFormatElement = null;
        vertexFormatIndex = 0;
        quadSprite = null;

        if (SmartAnimations.isActive()) {
            animatedSprites = animatedSprites == null ? animatedSpritesCached : animatedSprites;
            animatedSprites.clear();
        } else {
            animatedSprites = null;
        }
        modeTriangles = false;
    }

    public void begin(int glMode, VertexFormat format) {
        if (isDrawing) throw new IllegalStateException("Already building!");
        if (format == null) throw new IllegalArgumentException("VertexFormat cannot be null");
        isDrawing = true;
        reset();
        drawMode = glMode;
        vertexFormat = format;
        vertexFormatElement = format.getElement(0);
        noColor = false;
        byteBuffer.limit(byteBuffer.capacity());

        if (Config.isShaders()) SVertexBuilder.endSetVertexFormat(this);

        if (Config.isMultiTexture() && blockLayer != null) {
            quadSprites = quadSprites == null ? quadSpritesPrev : quadSprites;
            if (quadSprites == null || quadSprites.length < getBufferQuadSize()) {
                quadSprites = new TextureAtlasSprite[getBufferQuadSize()];
            }
        } else {
            quadSpritesPrev = quadSprites;
            quadSprites = null;
        }
    }

    public WorldRenderer tex(double u, double v) {
        if (vertexFormatElement == null) return this;
        if (quadSprite != null && quadSprites != null) {
            u = quadSprite.toSingleU((float)u);
            v = quadSprite.toSingleV((float)v);
            quadSprites[vertexCount >> 2] = quadSprite;
        }

        int offset = vertexCount * vertexFormat.getNextOffset() + vertexFormat.getOffset(vertexFormatIndex);
        switch (vertexFormatElement.getType()) {
            case FLOAT -> {
                byteBuffer.putFloat(offset, (float)u);
                byteBuffer.putFloat(offset + 4, (float)v);
            }
            case UINT, INT -> {
                byteBuffer.putInt(offset, (int)u);
                byteBuffer.putInt(offset + 4, (int)v);
            }
            case USHORT, SHORT -> {
                byteBuffer.putShort(offset, (short)v);
                byteBuffer.putShort(offset + 2, (short)u);
            }
            case UBYTE, BYTE -> {
                byteBuffer.put(offset, (byte)v);
                byteBuffer.put(offset + 1, (byte)u);
            }
        }
        nextVertexFormatIndex();
        return this;
    }

    public WorldRenderer lightmap(int light1, int light2) {
        if (vertexFormatElement == null) return this;
        int offset = vertexCount * vertexFormat.getNextOffset() + vertexFormat.getOffset(vertexFormatIndex);
        switch (vertexFormatElement.getType()) {
            case FLOAT -> {
                byteBuffer.putFloat(offset, light1);
                byteBuffer.putFloat(offset + 4, light2);
            }
            case UINT, INT -> {
                byteBuffer.putInt(offset, light1);
                byteBuffer.putInt(offset + 4, light2);
            }
            case USHORT, SHORT -> {
                byteBuffer.putShort(offset, (short)light2);
                byteBuffer.putShort(offset + 2, (short)light1);
            }
            case UBYTE, BYTE -> {
                byteBuffer.put(offset, (byte)light2);
                byteBuffer.put(offset + 1, (byte)light1);
            }
        }
        nextVertexFormatIndex();
        return this;
    }

    public void putBrightness4(int b1, int b2, int b3, int b4) {
        if (vertexCount < 4 || vertexFormat == null) return;
        int offset = (vertexCount - 4) * vertexFormat.getIntegerSize() + (vertexFormat.getUvOffsetById(1) >> 2);
        int stride = vertexFormat.getNextOffset() >> 2;
        rawIntBuffer.put(offset, b1);
        rawIntBuffer.put(offset + stride, b2);
        rawIntBuffer.put(offset + stride * 2, b3);
        rawIntBuffer.put(offset + stride * 3, b4);
    }

    public void putPosition(double x, double y, double z) {
        if (vertexCount < 4 || vertexFormat == null) return;
        int stride = vertexFormat.getIntegerSize();
        int base = (vertexCount - 4) * stride;
        float fx = (float)(x + xOffset);
        float fy = (float)(y + yOffset);
        float fz = (float)(z + zOffset);

        for (int i = 0; i < 4; i++) {
            int offset = base + i * stride;
            rawIntBuffer.put(offset, Float.floatToRawIntBits(fx + Float.intBitsToFloat(rawIntBuffer.get(offset))));
            rawIntBuffer.put(offset + 1, Float.floatToRawIntBits(fy + Float.intBitsToFloat(rawIntBuffer.get(offset + 1))));
            rawIntBuffer.put(offset + 2, Float.floatToRawIntBits(fz + Float.intBitsToFloat(rawIntBuffer.get(offset + 2))));
        }
    }

    public int getColorIndex(int vertex) {
        return vertexFormat != null
                ? ((vertexCount - vertex) * vertexFormat.getNextOffset() + vertexFormat.getColorOffset()) >> 2
                : 0;
    }

    public void putColorMultiplier(float r, float g, float b, int vertex) {
        int index = getColorIndex(vertex);
        if (!noColor && index >= 0 && index < rawIntBuffer.capacity()) {
            int color = rawIntBuffer.get(index);
            boolean littleEndian = ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN;
            int newColor = littleEndian
                    ? ((int)((color & 255) * r)) | ((int)((color >> 8 & 255) * g) << 8) |
                    ((int)((color >> 16 & 255) * b) << 16) | (color & 0xFF000000)
                    : ((int)((color >> 24 & 255) * r) << 24) | ((int)((color >> 16 & 255) * g) << 16) |
                    ((int)((color >> 8 & 255) * b) << 8) | (color & 255);
            rawIntBuffer.put(index, newColor);
        }
    }

    public void putColor(int argb, int vertex) {
        int index = getColorIndex(vertex);
        if (index >= 0 && index < rawIntBuffer.capacity()) {
            putColorRGBA(index, argb >> 16 & 255, argb >> 8 & 255, argb & 255, argb >> 24 & 255);
        }
    }

    public void putColorRGB_F(float r, float g, float b, int vertex) {
        int index = getColorIndex(vertex);
        if (index >= 0 && index < rawIntBuffer.capacity()) {
            putColorRGBA(index,
                    MathHelper.clamp_int((int)(r * 255), 0, 255),
                    MathHelper.clamp_int((int)(g * 255), 0, 255),
                    MathHelper.clamp_int((int)(b * 255), 0, 255), 255);
        }
    }

    public void putColorRGBA(int index, int r, int g, int b, int a) {
        if (index >= 0 && index < rawIntBuffer.capacity()) {
            rawIntBuffer.put(index, ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN
                    ? a << 24 | b << 16 | g << 8 | r
                    : r << 24 | g << 16 | b << 8 | a);
        }
    }

    public void noColor() {
        noColor = true;
    }

    public WorldRenderer color(float r, float g, float b, float a) {
        return color((int)(r * 255), (int)(g * 255), (int)(b * 255), (int)(a * 255));
    }

    public WorldRenderer color(int r, int g, int b, int a) {
        if (noColor || vertexFormatElement == null) return this;

        int offset = vertexCount * vertexFormat.getNextOffset() + vertexFormat.getOffset(vertexFormatIndex);
        switch (vertexFormatElement.getType()) {
            case FLOAT -> {
                byteBuffer.putFloat(offset, r / 255f);
                byteBuffer.putFloat(offset + 4, g / 255f);
                byteBuffer.putFloat(offset + 8, b / 255f);
                byteBuffer.putFloat(offset + 12, a / 255f);
            }
            case UINT, INT -> {
                byteBuffer.putInt(offset, r);
                byteBuffer.putInt(offset + 4, g);
                byteBuffer.putInt(offset + 8, b);
                byteBuffer.putInt(offset + 12, a);
            }
            case USHORT, SHORT -> {
                byteBuffer.putShort(offset, (short)r);
                byteBuffer.putShort(offset + 2, (short)g);
                byteBuffer.putShort(offset + 4, (short)b);
                byteBuffer.putShort(offset + 6, (short)a);
            }
            case UBYTE, BYTE -> {
                if (ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN) {
                    byteBuffer.put(offset, (byte)r);
                    byteBuffer.put(offset + 1, (byte)g);
                    byteBuffer.put(offset + 2, (byte)b);
                    byteBuffer.put(offset + 3, (byte)a);
                } else {
                    byteBuffer.put(offset, (byte)a);
                    byteBuffer.put(offset + 1, (byte)b);
                    byteBuffer.put(offset + 2, (byte)g);
                    byteBuffer.put(offset + 3, (byte)r);
                }
            }
        }
        nextVertexFormatIndex();
        return this;
    }

    public void addVertexData(int[] vertexData) {
        if (vertexData == null || vertexFormat == null) return;
        if (Config.isShaders()) SVertexBuilder.beginAddVertexData(this, vertexData);
        growBuffer(vertexData.length);
        rawIntBuffer.position(getBufferSize());
        rawIntBuffer.put(vertexData);
        vertexCount += vertexData.length / vertexFormat.getIntegerSize();
        if (Config.isShaders()) SVertexBuilder.endAddVertexData(this);
    }

    public void endVertex() {
        if (vertexFormat == null) return;
        vertexCount++;
        growBuffer(vertexFormat.getIntegerSize());
        vertexFormatIndex = 0;
        vertexFormatElement = vertexFormat.getElement(0);
        if (Config.isShaders()) SVertexBuilder.endAddVertex(this);
    }

    public WorldRenderer pos(double x, double y, double z) {
        if (vertexFormatElement == null) return this;
        if (Config.isShaders()) SVertexBuilder.beginAddVertex(this);

        int offset = vertexCount * vertexFormat.getNextOffset() + vertexFormat.getOffset(vertexFormatIndex);
        float fx = (float)(x + xOffset);
        float fy = (float)(y + yOffset);
        float fz = (float)(z + zOffset);

        switch (vertexFormatElement.getType()) {
            case FLOAT -> {
                byteBuffer.putFloat(offset, fx);
                byteBuffer.putFloat(offset + 4, fy);
                byteBuffer.putFloat(offset + 8, fz);
            }
            case UINT, INT -> {
                byteBuffer.putInt(offset, Float.floatToRawIntBits(fx));
                byteBuffer.putInt(offset + 4, Float.floatToRawIntBits(fy));
                byteBuffer.putInt(offset + 8, Float.floatToRawIntBits(fz));
            }
            case USHORT, SHORT -> {
                byteBuffer.putShort(offset, (short)fx);
                byteBuffer.putShort(offset + 2, (short)fy);
                byteBuffer.putShort(offset + 4, (short)fz);
            }
            case UBYTE, BYTE -> {
                byteBuffer.put(offset, (byte)fx);
                byteBuffer.put(offset + 1, (byte)fy);
                byteBuffer.put(offset + 2, (byte)fz);
            }
        }
        nextVertexFormatIndex();
        return this;
    }

    public void putNormal(float x, float y, float z) {
        if (vertexCount < 4 || vertexFormat == null) return;
        int packed = ((byte)(x * 127) & 255) | (((byte)(y * 127) & 255) << 8) | (((byte)(z * 127) & 255) << 16);
        int stride = vertexFormat.getNextOffset() >> 2;
        int offset = (vertexCount - 4) * stride + (vertexFormat.getNormalOffset() >> 2);
        for (int i = 0; i < 4; i++) {
            rawIntBuffer.put(offset + i * stride, packed);
        }
    }

    private void nextVertexFormatIndex() {
        if (vertexFormat == null) return;
        vertexFormatIndex = (vertexFormatIndex + 1) % vertexFormat.getElementCount();
        vertexFormatElement = vertexFormat.getElement(vertexFormatIndex);
        if (vertexFormatElement.getUsage() == VertexFormatElement.EnumUsage.PADDING) {
            nextVertexFormatIndex();
        }
    }

    public WorldRenderer normal(float x, float y, float z) {
        if (vertexFormatElement == null) return this;
        int offset = vertexCount * vertexFormat.getNextOffset() + vertexFormat.getOffset(vertexFormatIndex);
        switch (vertexFormatElement.getType()) {
            case FLOAT -> {
                byteBuffer.putFloat(offset, x);
                byteBuffer.putFloat(offset + 4, y);
                byteBuffer.putFloat(offset + 8, z);
            }
            case UINT, INT -> {
                byteBuffer.putInt(offset, (int)x);
                byteBuffer.putInt(offset + 4, (int)y);
                byteBuffer.putInt(offset + 8, (int)z);
            }
            case USHORT, SHORT -> {
                byteBuffer.putShort(offset, (short)(x * 32767));
                byteBuffer.putShort(offset + 2, (short)(y * 32767));
                byteBuffer.putShort(offset + 4, (short)(z * 32767));
            }
            case UBYTE, BYTE -> {
                byteBuffer.put(offset, (byte)(x * 127));
                byteBuffer.put(offset + 1, (byte)(y * 127));
                byteBuffer.put(offset + 2, (byte)(z * 127));
            }
        }
        nextVertexFormatIndex();
        return this;
    }

    public void setTranslation(double x, double y, double z) {
        xOffset = x;
        yOffset = y;
        zOffset = z;
    }

    public void finishDrawing() {
        if (!isDrawing) throw new IllegalStateException("Not building!");
        isDrawing = false;
        byteBuffer.position(0);
        byteBuffer.limit(getBufferSize() << 2);
    }

    public ByteBuffer getByteBuffer() {
        return modeTriangles ? byteBufferTriangles : byteBuffer;
    }

    public VertexFormat getVertexFormat() {
        return vertexFormat;
    }

    public int getVertexCount() {
        return modeTriangles ? (vertexCount / 4 * 6) : vertexCount;
    }

    public int getDrawMode() {
        return modeTriangles ? 4 : drawMode;
    }

    public void putColor4(int argb) {
        for (int i = 1; i <= 4; i++) putColor(argb, i);
    }

    public void putColorRGB_F4(float r, float g, float b) {
        for (int i = 1; i <= 4; i++) putColorRGB_F(r, g, b, i);
    }

    public void putSprite(TextureAtlasSprite sprite) {
        if (animatedSprites != null && sprite != null && sprite.getAnimationIndex() >= 0) {
            animatedSprites.set(sprite.getAnimationIndex());
        }
        if (quadSprites != null && vertexCount >= 4) {
            quadSprites[vertexCount / 4 - 1] = sprite;
        }
    }

    public void setSprite(TextureAtlasSprite sprite) {
        if (animatedSprites != null && sprite != null && sprite.getAnimationIndex() >= 0) {
            animatedSprites.set(sprite.getAnimationIndex());
        }
        quadSprite = sprite;
    }

    public boolean isMultiTexture() {
        return quadSprites != null;
    }

    public void drawMultiTexture() {
        if (quadSprites == null) return;

        int maxSprites = Config.getMinecraft().getTextureMapBlocks().getCountRegisteredSprites();
        if (drawnIcons.length <= maxSprites) {
            drawnIcons = new boolean[maxSprites + 1];
        }
        Arrays.fill(drawnIcons, false);

        int quadCount = vertexCount >> 2;
        int grassSideIndex = -1;

        for (int i = 0; i < quadCount; i++) {
            TextureAtlasSprite sprite = quadSprites[i];
            if (sprite != null) {
                int spriteIndex = sprite.getIndexInMap();
                if (spriteIndex >= 0 && spriteIndex < drawnIcons.length && !drawnIcons[spriteIndex]) {
                    if (sprite == TextureUtils.iconGrassSideOverlay) {
                        if (grassSideIndex < 0) grassSideIndex = i;
                    } else {
                        i = drawForIcon(sprite, i) - 1;
                        if (blockLayer != EnumWorldBlockLayer.TRANSLUCENT) {
                            drawnIcons[spriteIndex] = true;
                        }
                    }
                }
            }
        }

        if (grassSideIndex >= 0) drawForIcon(TextureUtils.iconGrassSideOverlay, grassSideIndex);
    }

    private int drawForIcon(TextureAtlasSprite sprite, int start) {
        if (sprite == null) return start;
        GL11.glBindTexture(GL11.GL_TEXTURE_2D, sprite.glSpriteTextureId);
        int quadCount = vertexCount >> 2;
        int firstMatch = -1;
        int lastMatch = -1;

        for (int i = start; i < quadCount; i++) {
            if (quadSprites[i] == sprite) {
                if (firstMatch < 0) firstMatch = i;
            } else if (firstMatch >= 0) {
                draw(firstMatch, i);
                if (blockLayer == EnumWorldBlockLayer.TRANSLUCENT) return i;
                firstMatch = -1;
                if (lastMatch < 0) lastMatch = i;
            }
        }

        if (firstMatch >= 0) draw(firstMatch, quadCount);
        return lastMatch < 0 ? quadCount : lastMatch;
    }

    private void draw(int start, int end) {
        int count = (end - start) << 2;
        if (count > 0) GL11.glDrawArrays(drawMode, start << 2, count);
    }

    public void setBlockLayer(EnumWorldBlockLayer layer) {
        blockLayer = layer;
        if (layer == null) {
            quadSpritesPrev = quadSprites;
            quadSprites = null;
            quadSprite = null;
        }
    }

    public int getBufferQuadSize() {
        return vertexFormat != null ? rawIntBuffer.capacity() / vertexFormat.getIntegerSize() : 0;
    }

    public RenderEnv getRenderEnv(IBlockState state, BlockPos pos) {
        if (state == null || pos == null) throw new IllegalArgumentException("State and position cannot be null");
        if (renderEnv == null) {
            renderEnv = new RenderEnv(state, pos);
        } else {
            renderEnv.reset(state, pos);
        }
        return renderEnv;
    }

    public boolean isDrawing() {
        return isDrawing;
    }

    public double getXOffset() {
        return xOffset;
    }

    public double getYOffset() {
        return yOffset;
    }

    public double getZOffset() {
        return zOffset;
    }

    public EnumWorldBlockLayer getBlockLayer() {
        return blockLayer;
    }

    public void putColorMultiplierRgba(float r, float g, float b, float a, int vertex) {
        int index = getColorIndex(vertex);
        if (!noColor && index >= 0 && index < rawIntBuffer.capacity()) {
            int color = rawIntBuffer.get(index);
            boolean littleEndian = ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN;
            int newColor = littleEndian
                    ? ((int)((color & 255) * r)) | ((int)((color >> 8 & 255) * g) << 8) |
                    ((int)((color >> 16 & 255) * b) << 16) | ((int)((color >>> 24) * a) << 24)
                    : ((int)((color >>> 24) * r) << 24) | ((int)((color >> 16 & 255) * g) << 16) |
                    ((int)((color >> 8 & 255) * b) << 8) | ((int)((color & 255) * a));
            rawIntBuffer.put(index, newColor);
        }
    }

    public void quadsToTriangles() {
        if (drawMode != 7 || vertexFormat == null) return;

        if (byteBufferTriangles == null || byteBufferTriangles.capacity() < (byteBuffer.capacity() << 1)) {
            byteBufferTriangles = GLAllocation.createDirectByteBuffer(byteBuffer.capacity() << 1);
        }

        int stride = vertexFormat.getNextOffset();
        byteBuffer.rewind();
        byteBufferTriangles.clear();

        for (int i = 0; i < vertexCount; i += 4) {
            byteBuffer.limit((i + 3) * stride);
            byteBuffer.position(i * stride);
            byteBufferTriangles.put(byteBuffer);
            byteBuffer.limit((i + 1) * stride);
            byteBuffer.position(i * stride);
            byteBufferTriangles.put(byteBuffer);
            byteBuffer.limit((i + 4) * stride);
            byteBuffer.position((i + 2) * stride);
            byteBufferTriangles.put(byteBuffer);
        }

        byteBuffer.rewind();
        byteBufferTriangles.flip();
        modeTriangles = true;
    }

    public boolean isColorDisabled() {
        return noColor;
    }
}