package net.minecraft.client.main;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.mojang.authlib.properties.PropertyMap;
import com.mojang.authlib.properties.PropertyMap.Serializer;

import java.io.File;
import java.net.Authenticator;
import java.net.InetSocketAddress;
import java.net.PasswordAuthentication;
import java.net.Proxy;
import java.net.Proxy.Type;
import java.util.List;

import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import net.minecraft.client.Minecraft;
import net.minecraft.util.Session;
import org.apache.commons.math3.analysis.function.Min;
import tech.atani.client.Atani;

public class Main {
    public static void main(String[] args) {
        System.setProperty("java.net.preferIPv4Stack", "true");
        
        OptionParser parser = new OptionParser();
        parser.allowsUnrecognizedOptions();

        parser.accepts("fullscreen");
        parser.accepts("checkGlErrors");

        OptionSpec<String> serverHostArg = parser.accepts("server").withRequiredArg();
        OptionSpec<Integer> serverPortArg = parser.accepts("port").withRequiredArg().ofType(Integer.class).defaultsTo(25565);
        OptionSpec<File> gameDirArg = parser.accepts("gameDir").withRequiredArg().ofType(File.class).defaultsTo(new File("."));
        OptionSpec<File> assetsDirArg = parser.accepts("assetsDir").withRequiredArg().ofType(File.class);
        OptionSpec<File> resourcePackDirArg = parser.accepts("resourcePackDir").withRequiredArg().ofType(File.class);
        OptionSpec<String> proxyHostArg = parser.accepts("proxyHost").withRequiredArg();
        OptionSpec<Integer> proxyPortArg = parser.accepts("proxyPort").withRequiredArg().defaultsTo("8080", new String[0]).ofType(Integer.class);
        OptionSpec<String> proxyUsernameArg = parser.accepts("proxyUser").withRequiredArg();
        OptionSpec<String> proxyPassArg = parser.accepts("proxyPass").withRequiredArg();
        OptionSpec<String> usernameArg = parser.accepts("username").withRequiredArg().defaultsTo("Player" + Minecraft.getSystemTime() % 1000L);
        OptionSpec<String> uuidArg = parser.accepts("uuid").withRequiredArg();
        OptionSpec<String> accessTokenArg = parser.accepts("accessToken").withRequiredArg().required();
        OptionSpec<String> versionArg = parser.accepts("version").withRequiredArg().required();
        OptionSpec<Integer> widthArg = parser.accepts("width").withRequiredArg().ofType(Integer.class).defaultsTo(854);
        OptionSpec<Integer> heightArg = parser.accepts("height").withRequiredArg().ofType(Integer.class).defaultsTo(480);
        OptionSpec<String> profilePropertyArg = parser.accepts("profileProperties").withRequiredArg().defaultsTo("{}");
        OptionSpec<String> assetIndexArg = parser.accepts("assetIndex").withRequiredArg();
        OptionSpec<String> userTypeArg = parser.accepts("userType").withRequiredArg().defaultsTo("legacy");

        OptionSpec<String> nonOptions = parser.nonOptions();
        OptionSet set = parser.parse(args);
        List<String> list = set.valuesOf(nonOptions);

        if (!list.isEmpty()) {
            System.out.println("Completely ignored arguments: " + list);
        }

        String proxyHost = set.valueOf(proxyHostArg);
        Proxy proxy = Proxy.NO_PROXY;

        if (proxyHost != null) {
            try {
                proxy = new Proxy(Type.SOCKS, new InetSocketAddress(proxyHost, set.valueOf(proxyPortArg)));
            } catch (Exception ignored) {
            }
        }

        final String proxyUsername = set.valueOf(proxyUsernameArg);
        final String proxyPassword = set.valueOf(proxyPassArg);

        if (!proxy.equals(Proxy.NO_PROXY) && isNullOrEmpty(proxyUsername) && isNullOrEmpty(proxyPassword)) {
            Authenticator.setDefault(new Authenticator() {
                protected PasswordAuthentication getPasswordAuthentication() {
                    return new PasswordAuthentication(proxyUsername, proxyPassword.toCharArray());
                }
            });
        }

        int screenWidth = set.valueOf(widthArg);
        int screenHeight = set.valueOf(heightArg);

        boolean fullscreen = set.has("fullscreen");
        boolean checkGLErrors = set.has("checkGlErrors");

        String version = set.valueOf(versionArg);
        Gson gson = (new GsonBuilder()).registerTypeAdapter(PropertyMap.class, new Serializer()).create();
        PropertyMap profilePropertyMap = gson.fromJson(set.valueOf(profilePropertyArg), PropertyMap.class);
        
        File gameDir = set.valueOf(gameDirArg);
        File assetsDir = set.has(assetsDirArg) ? set.valueOf(assetsDirArg) : new File(gameDir, "assets/");
        File resourcePacksDir = set.has(resourcePackDirArg) ? set.valueOf(resourcePackDirArg) : new File(gameDir, "resourcepacks/");
        
        String uuid = set.has(uuidArg) ? uuidArg.value(set) : usernameArg.value(set);
        String assetIndex = set.has(assetIndexArg) ? assetIndexArg.value(set) : null;

        String serverIP = set.valueOf(serverHostArg);
        Integer serverPort = set.valueOf(serverPortArg);

        Session session = new Session(usernameArg.value(set), uuid, accessTokenArg.value(set), userTypeArg.value(set));
        GameConfiguration configuration = new GameConfiguration(
                new GameConfiguration.UserInformation(
                        session,
                        profilePropertyMap,
                        proxy
                ),
                new GameConfiguration.DisplayInformation(
                        screenWidth,
                        screenHeight,
                        fullscreen,
                        checkGLErrors
                ),
                new GameConfiguration.FolderInformation(
                        gameDir,
                        resourcePacksDir,
                        assetsDir,
                        assetIndex
                ),
                new GameConfiguration.GameInformation(
                        version
                ),
                new GameConfiguration.ServerInformation(
                        serverIP,
                        serverPort
                )
        );

        Thread shutdownHook = new Thread(Minecraft::stopIntegratedServer);
        shutdownHook.setName("Client Shutdown Thread");
        Runtime.getRuntime().addShutdownHook(shutdownHook);

        Thread.currentThread().setName("Client thread");
        new Minecraft(configuration).run();
    }

    private static boolean isNullOrEmpty(String string) {
        return string != null && !string.isEmpty();
    }
}
