package net.minecraft.client.audio;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import io.netty.util.internal.ThreadLocalRandom;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import net.minecraft.client.Minecraft;
import net.minecraft.client.resources.IResourceManager;
import net.minecraft.client.resources.IResourceManagerReloadListener;
import net.minecraft.client.settings.GameSettings;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.util.MathHelper;
import net.minecraft.util.ResourceLocation;
import org.apache.commons.io.IOUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.Marker;
import org.apache.logging.log4j.MarkerManager;
import org.lwjgl.openal.*;

import static org.lwjgl.openal.AL10.*;

public class SoundManager implements IResourceManagerReloadListener {
    private static final Marker LOG_MARKER = MarkerManager.getMarker("SOUNDS");
    private static final Logger logger = LogManager.getLogger();
    private final SoundHandler sndHandler;
    private final GameSettings options;
    private SoundSystemStarterThread sndSystem;
    private boolean loaded;
    private int playTime = 0;
    private final Map<String, ISound> playingSounds = HashBiMap.create();
    private final Map<ISound, String> invPlayingSounds = ((BiMap<String, ISound>) playingSounds).inverse();
    private final Map<ISound, SoundPoolEntry> playingSoundPoolEntries = Maps.newHashMap();
    private final Multimap<SoundCategory, String> categorySounds = HashMultimap.create();
    private final List<ITickableSound> tickableSounds = Lists.newArrayList();
    private final Map<ISound, Integer> delayedSounds = Maps.newHashMap();
    private final Map<String, Integer> playingSoundsStopTime = Maps.newHashMap();
    private final Map<String, Integer> soundBuffers = new HashMap<>();

    public SoundManager(SoundHandler soundHandler, GameSettings gameSettings) {
        this.sndHandler = soundHandler;
        this.options = gameSettings;
    }

    public void reloadSoundSystem() {
        unloadSoundSystem();
        loadSoundSystem();
    }

    private synchronized void loadSoundSystem() {
        if (!loaded) {
            try {
                SoundSystemOpenAL.create();
                sndSystem = new SoundSystemStarterThread();
                loaded = true;
                sndSystem.setMasterVolume(options.getSoundLevel(SoundCategory.MASTER));
            } catch (Exception e) {
                logger.error(LOG_MARKER, "Error starting SoundSystem", e);
                options.setSoundLevel(SoundCategory.MASTER, 0.0F);
                options.saveOptions();
            }
        }
    }

    private float getSoundCategoryVolume(SoundCategory category) {
        return category != null && category != SoundCategory.MASTER ? options.getSoundLevel(category) : 1.0F;
    }

    public void setSoundCategoryVolume(SoundCategory category, float volume) {
        if (loaded && category != null) {
            if (category == SoundCategory.MASTER) {
                sndSystem.setMasterVolume(volume);
            } else {
                for (String s : categorySounds.get(category)) {
                    ISound isound = playingSounds.get(s);
                    if (isound != null) {
                        float f = getNormalizedVolume(isound, playingSoundPoolEntries.get(isound), category);
                        sndSystem.setVolume(s, f);
                        if (f <= 0.0F) stopSound(isound);
                    }
                }
            }
        }
    }

    public void unloadSoundSystem() {
        if (loaded) {
            stopAllSounds();
            synchronized (soundBuffers) {
                for (Integer sourceId : sndSystem.soundSources.values()) {
                    if (alIsSource(sourceId)) {
                        int bufferId = alGetSourcei(sourceId, AL_BUFFER);
                        if (bufferId != 0) {
                            alSourceStop(sourceId);
                            alSourcei(sourceId, AL_BUFFER, 0);
                        }
                    }
                }
                for (Integer bufferId : soundBuffers.values()) {
                    if (bufferId != null && alIsBuffer(bufferId)) alDeleteBuffers(bufferId);
                }
                soundBuffers.clear();
            }
            sndSystem.cleanup();
            loaded = false;
        }
    }

    public void stopAllSounds() {
        if (loaded) {
            for (String s : playingSounds.keySet()) sndSystem.stop(s);
            playingSounds.clear();
            delayedSounds.clear();
            tickableSounds.clear();
            categorySounds.clear();
            playingSoundPoolEntries.clear();
            playingSoundsStopTime.clear();
        }
    }

    public void updateAllSounds() {
        if (!loaded) return;
        ++playTime;
        Minecraft mc = Minecraft.getMinecraft();
        boolean isGamePaused = mc.currentScreen != null && mc.isGamePaused();
        Iterator<ITickableSound> tickableIterator = tickableSounds.iterator();
        while (tickableIterator.hasNext()) {
            ITickableSound itickablesound = tickableIterator.next();
            itickablesound.update();
            if (itickablesound.isDonePlaying()) {
                stopSound(itickablesound);
                tickableIterator.remove();
            } else if (!isGamePaused) {
                String s = invPlayingSounds.get(itickablesound);
                if (s != null && sndSystem.playing(s)) {
                    SoundEventAccessorComposite soundEvent = sndHandler.getSound(itickablesound.getSoundLocation());
                    if (soundEvent != null) {
                        sndSystem.setVolume(s, getNormalizedVolume(itickablesound, playingSoundPoolEntries.get(itickablesound), soundEvent.getSoundCategory()));
                        sndSystem.setPitch(s, getNormalizedPitch(itickablesound, playingSoundPoolEntries.get(itickablesound)));
                        sndSystem.setPosition(s, itickablesound.getXPosF(), itickablesound.getYPosF(), itickablesound.getZPosF());
                    }
                }
            }
        }
        Iterator<Entry<String, ISound>> iterator = playingSounds.entrySet().iterator();
        while (iterator.hasNext()) {
            Entry<String, ISound> entry = iterator.next();
            String s1 = entry.getKey();
            ISound isound = entry.getValue();
            Integer sourceId = sndSystem.soundSources.get(s1);
            if (sourceId != null) {
                int state = alGetSourcei(sourceId, AL_SOURCE_STATE);
                if (state == AL_STOPPED) {
                    int i = playingSoundsStopTime.getOrDefault(s1, 0);
                    if (i <= playTime) {
                        if (isound.canRepeat() && isound.getRepeatDelay() > 0) delayedSounds.put(isound, playTime + isound.getRepeatDelay());
                        iterator.remove();
                        playingSoundPoolEntries.remove(isound);
                        playingSoundsStopTime.remove(s1);
                        SoundEventAccessorComposite soundEvent = sndHandler.getSound(isound.getSoundLocation());
                        if (soundEvent != null) categorySounds.remove(soundEvent.getSoundCategory(), s1);
                        if (isound instanceof ITickableSound) tickableSounds.remove(isound);
                        sndSystem.removeSource(s1);
                    }
                }
            }
        }
        Iterator<Entry<ISound, Integer>> delayedIterator = delayedSounds.entrySet().iterator();
        while (delayedIterator.hasNext()) {
            Entry<ISound, Integer> entry = delayedIterator.next();
            if (playTime >= entry.getValue()) {
                ISound isound = entry.getKey();
                if (isound instanceof ITickableSound) ((ITickableSound) isound).update();
                playSound(isound);
                delayedIterator.remove();
            }
        }
    }

    public boolean isSoundPlaying(ISound sound) {
        if (!loaded || sound == null) return false;
        String s = invPlayingSounds.get(sound);
        return s != null && sndSystem.playing(s);
    }

    public void stopSound(ISound sound) {
        if (loaded && sound != null) {
            String s = invPlayingSounds.get(sound);
            if (s != null) {
                sndSystem.stop(s);
                sndSystem.removeSource(s);
                playingSounds.remove(s);
                playingSoundPoolEntries.remove(sound);
                playingSoundsStopTime.remove(s);
                SoundEventAccessorComposite soundEvent = sndHandler.getSound(sound.getSoundLocation());
                if (soundEvent != null) categorySounds.remove(soundEvent.getSoundCategory(), s);
                if (sound instanceof ITickableSound) tickableSounds.remove(sound);
            }
        }
    }

    public void playSoundFromStream(InputStream inputStream, float volume, float pitch, String sourceIdentifier) {
        if (!loaded || sndSystem.getMasterVolume() <= 0.0F || inputStream == null) {
            IOUtils.closeQuietly(inputStream);
            return;
        }
        try {
            byte[] soundBytes = IOUtils.toByteArray(inputStream);
            ByteBuffer vorbisData = ByteBuffer.allocateDirect(soundBytes.length).put(soundBytes);
            vorbisData.flip();
            int bufferId = OggDecoder.loadOgg(vorbisData);
            if (bufferId == 0) {
                logger.error(LOG_MARKER, "Failed to load Ogg data etc {}", sourceIdentifier);
                return;
            }
            String sourceName = MathHelper.getRandomUuid(ThreadLocalRandom.current()).toString();
            sndSystem.newSource(false, sourceName, null, sourceIdentifier, false, 0.0F, 0.0F, 0.0F, ISound.AttenuationType.NONE.getTypeInt(), volume, options.reverb);
            Integer sourceId = sndSystem.soundSources.get(sourceName);
            if (sourceId != null && AL10.alIsSource(sourceId)) {
                AL10.alSourcei(sourceId, AL10.AL_BUFFER, bufferId);
                AL10.alSourcef(sourceId, AL10.AL_PITCH, Math.max(0.5f, Math.min(pitch, 2.0f)));
                AL10.alSourcef(sourceId, AL10.AL_GAIN, Math.max(0.0f, Math.min(volume * getSoundCategoryVolume(SoundCategory.MASTER), 1.0f)));
                AL10.alSourcePlay(sourceId);
                playingSoundsStopTime.put(sourceName, playTime + 20);
            } else {
                alDeleteBuffers(bufferId);
            }
        } catch (Exception e) {
            logger.error(LOG_MARKER, "Error playing sound from stream with identifier: {}", sourceIdentifier, e);
        } finally {
            IOUtils.closeQuietly(inputStream);
        }
    }

    public void playSound(ISound p_sound) {
        if (!loaded || sndSystem.getMasterVolume() <= 0.0F || p_sound == null) return;
        SoundEventAccessorComposite soundEvent = sndHandler.getSound(p_sound.getSoundLocation());
        if (soundEvent == null || soundEvent.cloneEntry() == SoundHandler.missing_sound) return;
        SoundPoolEntry soundEntry = soundEvent.cloneEntry();
        String sourceName = MathHelper.getRandomUuid(ThreadLocalRandom.current()).toString();
        ResourceLocation bufferLocation = soundEntry.getSoundPoolEntryLocation();
        URL soundURL = getURLForSoundResource(bufferLocation);
        Integer bufferId;
        synchronized (soundBuffers) {
            bufferId = soundBuffers.get(bufferLocation.toString());
            if (bufferId == null || !AL10.alIsBuffer(bufferId)) {
                try (InputStream is = soundURL.openStream()) {
                    byte[] soundBytes = IOUtils.toByteArray(is);
                    ByteBuffer vorbisData = ByteBuffer.allocateDirect(soundBytes.length * 2).put(soundBytes);
                    vorbisData.flip();
                    bufferId = OggDecoder.loadOgg(vorbisData);
                    if (bufferId == 0) {
                        logger.error(LOG_MARKER, "Failed to load Ogg data for {}", bufferLocation);
                        return;
                    }
                    soundBuffers.put(bufferLocation.toString(), bufferId);
                } catch (Exception e) {
                    logger.error(LOG_MARKER, "Error loading sound buffer: {}", bufferLocation, e);
                    return;
                }
            }
        }
        float volume = getNormalizedVolume(p_sound, soundEntry, soundEvent.getSoundCategory());
        float pitch = getNormalizedPitch(p_sound, soundEntry);
        boolean looping = p_sound.canRepeat() && p_sound.getRepeatDelay() == 0;
        sndSystem.newSource(false, sourceName, soundURL, bufferLocation.toString(), looping, p_sound.getXPosF(), p_sound.getYPosF(), p_sound.getZPosF(), p_sound.getAttenuationType().getTypeInt(), volume, options.reverb);
        Integer sourceId = sndSystem.soundSources.get(sourceName);
        if (sourceId != null && AL10.alIsSource(sourceId)) {
            AL10.alSourcei(sourceId, AL10.AL_BUFFER, bufferId);
            AL10.alSourcef(sourceId, AL10.AL_PITCH, pitch);
            AL10.alSourcef(sourceId, AL10.AL_GAIN, volume);
            AL10.alSourcePlay(sourceId);
            playingSounds.put(sourceName, p_sound);
            playingSoundPoolEntries.put(p_sound, soundEntry);
            playingSoundsStopTime.put(sourceName, looping ? Integer.MAX_VALUE : playTime + 20);
            if (soundEvent.getSoundCategory() != SoundCategory.MASTER) categorySounds.put(soundEvent.getSoundCategory(), sourceName);
            if (p_sound instanceof ITickableSound) tickableSounds.add((ITickableSound) p_sound);
        }
    }

    private float getNormalizedPitch(ISound sound, SoundPoolEntry entry) {
        return (float) MathHelper.clamp_double(sound.getPitch() * entry.getPitch(), 0.5D, 2.0D);
    }

    private float getNormalizedVolume(ISound sound, SoundPoolEntry entry, SoundCategory category) {
        return (float) MathHelper.clamp_double(sound.getVolume() * entry.getVolume(), 0.0D, 1.0D) * getSoundCategoryVolume(category);
    }

    public void pauseAllSounds() {
        if (loaded) for (String s : playingSounds.keySet()) sndSystem.pause(s);
    }

    public void resumeAllSounds() {
        if (loaded) for (String s : playingSounds.keySet()) sndSystem.play(s);
    }

    public void playDelayedSound(ISound sound, int delay) {
        if (loaded && sound != null) delayedSounds.put(sound, playTime + Math.max(0, delay));
    }

    private static URL getURLForSoundResource(ResourceLocation location) {
        String s = String.format("%s:%s:%s", "mcsounddomain", location.getResourceDomain(), location.getResourcePath());
        URLStreamHandler urlstreamhandler = new URLStreamHandler() {
            protected URLConnection openConnection(URL url) {
                return new URLConnection(url) {
                    public void connect() {}
                    public InputStream getInputStream() throws java.io.IOException {
                        return Minecraft.getMinecraft().getResourceManager().getResource(location).getInputStream();
                    }
                };
            }
        };
        try {
            return new URL(null, s, urlstreamhandler);
        } catch (Exception e) {
            throw new RuntimeException("Failed to create sound URL", e);
        }
    }

    public void setListener(EntityPlayer player, float renderPartialTicks) {
        if (loaded && player != null) {
            float f = player.prevRotationPitch + (player.rotationPitch - player.prevRotationPitch) * renderPartialTicks;
            float f1 = player.prevRotationYaw + (player.rotationYaw - player.prevRotationYaw) * renderPartialTicks;
            double x = player.prevPosX + (player.posX - player.prevPosX) * renderPartialTicks;
            double y = player.prevPosY + (player.posY - player.prevPosY) * renderPartialTicks + player.getEyeHeight();
            double z = player.prevPosZ + (player.posZ - player.prevPosZ) * renderPartialTicks;
            float f2 = MathHelper.cos((f1 + 90.0F) * 0.017453292F);
            float f3 = MathHelper.sin((f1 + 90.0F) * 0.017453292F);
            float f4 = MathHelper.cos(-f * 0.017453292F);
            float atY = MathHelper.sin(-f * 0.017453292F);
            float f6 = MathHelper.cos((-f + 90.0F) * 0.017453292F);
            float upY = MathHelper.sin((-f + 90.0F) * 0.017453292F);
            float atX = f2 * f4;
            float atZ = f3 * f4;
            float upX = f2 * f6;
            float upZ = f3 * f6;
            sndSystem.setListenerPosition((float) x, (float) y, (float) z);
            sndSystem.setListenerOrientation(atX, atY, atZ, upX, upY, upZ);
        }
    }

    @Override
    public void onResourceManagerReload(IResourceManager resourceManager) {
        if (loaded) {
            stopAllSounds();
            synchronized (soundBuffers) {
                for (Integer sourceId : sndSystem.soundSources.values()) {
                    if (alIsSource(sourceId)) {
                        int bufferId = alGetSourcei(sourceId, AL_BUFFER);
                        if (bufferId != 0) {
                            alSourceStop(sourceId);
                            alSourcei(sourceId, AL_BUFFER, 0);
                        }
                    }
                }
                for (Integer bufferId : soundBuffers.values()) {
                    if (bufferId != null && alIsBuffer(bufferId)) alDeleteBuffers(bufferId);
                }
                soundBuffers.clear();
            }
        }
    }

    private void checkOpenALError(String operation) {
        int error = AL10.alGetError();
        if (error != AL_NO_ERROR) logger.error(LOG_MARKER, "OpenAL error during {}: {}", operation, error);
    }

    static class SoundSystemStarterThread extends SoundSystemOpenAL {}
}