package net.optifine.shaders.file

import com.google.common.base.Splitter
import com.google.gson.JsonObject
import net.bloom.bloomclient.file.FileConfig
import net.bloom.bloomclient.file.FileManager
import net.minecraft.client.MinecraftInstance
import net.bloom.bloomclient.value.Value
import net.bloom.bloomclient.value.values.GroupValue
import net.bloom.bloomclient.value.values.shaders.*
import net.minecraft.src.Config
import net.optifine.expr.ExpressionParser
import net.optifine.expr.IExpressionBool
import net.optifine.expr.ParseException
import net.optifine.shaders.SMCLog
import net.optifine.shaders.ShaderManager
import net.optifine.shaders.ShaderParserUtils
import net.optifine.shaders.Shaders
import net.optifine.shaders.config.MacroProcessor
import net.optifine.shaders.gui.*
import net.optifine.shaders.shaderoption.ShaderOption
import net.optifine.shaders.shaderoption.ShaderProfile
import net.optifine.shaders.shaderoption.ShaderResolver
import net.optifine.util.PropertiesOrdered
import java.io.File
import java.io.InputStream
import java.util.regex.Pattern

const val SHADER_PROPERTIES_PATH = "/shaders/shaders.properties"
const val SHADER_EN_US_LANG_PATH = "/shaders/lang/en_US.lang"

val splitter: Splitter = Splitter.on('=').limit(2)
val pattern: Pattern = Pattern.compile("%(\\d+\\$)?[\\d\\.]*[df]")
val programEnabled: Pattern = Pattern.compile("program\\.([^.]+)\\.enabled")

abstract class AbstractShaderPack(val name: String): MinecraftInstance() {
    val options = ShaderOption().also { it.initValues() }

    val langProperties = hashMapOf<String, String>()
    val dimensionList = mutableListOf<Int>()
    private val sliderNames = hashSetOf<String>()
    val properties = PropertiesOrdered()
    val programConditions = hashMapOf<String, IExpressionBool>()

    val config = object: FileConfig(File(ShaderManager.shaderConfigFolder, "$name.json")) {
        override fun fromJson(json: JsonObject) {
            for ((key, value) in json.entrySet())
                options.shaderValues.find { it.name.equals(key, true ) }?.fromJson(value)
        }

        override fun toJson(): JsonObject {
            val json = JsonObject()

            for (value in options.reducedValues)
                json.add(value.name, value.toJson())

            return json
        }
    }

    abstract fun getResourceAsStream(resourceName: String?): InputStream?

    abstract fun hasDirectory(name: String?): Boolean

    abstract fun close()

    fun load() {
        loadProperties()
        readSliders()
        loadLanguage()
        loadDimensions()
        loadShaderPackOptions()
        loadProfiles()
        loadGroup()
        loadProgramConditions()

        loadConfig()
        options.createComponentFromImportedValues(this)
    }

    private fun loadProperties() {
        val inputStream = getResourceAsStream(SHADER_PROPERTIES_PATH) ?: return
        val macroedInputStream = MacroProcessor.process(inputStream, SHADER_PROPERTIES_PATH)

        properties.load(macroedInputStream)
        macroedInputStream.close()
    }

    private fun readSliders() {
        properties.entries.forEach { (keyRaw, valueRaw) ->
            val key = keyRaw.toString()
            val value = valueRaw.toString()

            if (key.equals("sliders", true))
                sliderNames.addAll(value.split(" "))
        }
    }

    private fun loadConfig() {
        properties.entries.forEach { (keyRaw, valueRaw) ->
            val key = keyRaw.toString()
            val value = valueRaw.toString()

            options.globalGroup.value.find { it.name == key }?.fromProperties(value)
        }

        FileManager.loadConfig(config)
    }

    private fun loadProfiles() {
        val profiles = mutableSetOf<ShaderProfile>()

        properties.entries.forEach { (keyRaw, _) ->
            val key = keyRaw.toString()

            if (key.startsWith("profile.", true)) {
                val name = key.removePrefix("profile.")
                ShaderParserUtils.parseProfile(this, name, properties, mutableSetOf())?.let { profiles.add(it) }
            }
        }

        if (profiles.isNotEmpty())
            options.shaderValues.add(ShaderProfileValue("Profile", "Profile", CustomProfile, profiles))
    }

    private fun loadGroup() {
        properties.entries.forEach { (keyRaw, _) ->
            val key = keyRaw.toString()

            if (key.startsWith("screen.", true)) {
                ShaderParserUtils.parseGuiScreen(this, key, properties)?.let { options.shaderValues.add(it) }
            }
        }
    }

    fun loadLanguage() {
        langProperties.clear()

        val inputStream = getResourceAsStream(SHADER_EN_US_LANG_PATH) ?: return
        val lines = inputStream.bufferedReader().use { it.readLines() }.filter { it.isNotEmpty() && it[0].code != 35 }

        for (line in lines) {
            val stringArray = splitter.split(line).toList()
            if (stringArray.size == 2) {
                val key = stringArray[0]
                val value = pattern.matcher(stringArray[1]).replaceAll("%$1s")
                langProperties[key] = value
            }
        }
    }

    private fun loadDimensions() {
        dimensionList.clear()

        for (i in -128..128) {
            if (hasDirectory("/shaders/world$i"))
                dimensionList.add(i)
        }
    }

    private fun loadShaderPackOptions() {
        options.shaderValues.clear()

        val programs = Shaders.programs.programNames.filter { it.isNotEmpty() }
        val shaderOptions = hashMapOf<String, IShaderOption>()
        collectShaderOptions(programs, "/shaders", shaderOptions)

        for (i in dimensionList)
            collectShaderOptions(programs, "/shaders/world$i", shaderOptions)

        shaderOptions.values.filterIsInstance<Value<*>>().forEach { options.shaderValues.add(it) }
    }

    private fun collectShaderOptions(programs: List<String>, path: String, options: HashMap<String, IShaderOption>) {
        for (program in programs) {
            collectShaderOptions("$path/$program.fsh", options)
            collectShaderOptions("$path/$program.vsh", options)
        }
    }

    private fun collectShaderOptions(path: String, options: HashMap<String, IShaderOption>) {
        val lines = ShaderParserUtils.getLines(this, path)

        for (line in lines) {
            val iShaderOption = ShaderParserUtils.getShaderOptionFromLine(line, path, sliderNames) ?: continue

            if (iShaderOption is Value<*> && (!iShaderOption.checkUsed() || lines.any { iShaderOption.isUsedInLine(it) })) {
                val name = iShaderOption.name
                val mappedShaderOption = options[name]

                if (mappedShaderOption != null && mappedShaderOption is Value<*>) {
                    if (mappedShaderOption.defaultValue != iShaderOption.defaultValue) {
                        Config.warn("Detected ambiguous shader option: $name")
                        Config.warn(" - in ${mappedShaderOption.paths}: ${mappedShaderOption.defaultValue}")
                        Config.warn(" - in ${iShaderOption.paths}: ${iShaderOption.defaultValue}")
                        Config.warn("This value will override old value!!!")
                        options[name] = iShaderOption.also { it.paths.addAll(mappedShaderOption.paths) }
                    } else {
                        mappedShaderOption.paths.addAll(iShaderOption.paths)
                    }

                } else options[name] = iShaderOption
            }
        }
    }

    fun detectProfile(define: Boolean): ShaderProfile? {
        val profileValue = options.shaderValues.filterIsInstance<ShaderProfileValue>().firstOrNull() ?: return null

        for (profile in profileValue.profiles) {
            if (matchProfile(profile, define))
                return profile
        }

        return null
    }

    private fun matchProfile(profile: ShaderProfile, define: Boolean): Boolean {
        for ((keyProfile, valueProfile) in profile.options) {
            val optionValue = options.shaderValues.find { it.name == keyProfile } ?: continue
            val value = if (define) optionValue.defaultValue.toString() else optionValue.value.toString()

            if (value != valueProfile)
                return false
        }

        return true
    }

    private fun loadProgramConditions() {
        properties.entries.forEach { (keyRaw, valueRaw) ->
            val key = keyRaw.toString()
            val value = valueRaw.toString().trim()
            val matcher = programEnabled.matcher(key)
            if (matcher.matches()) {
                val programPath = matcher.group(1)
                val iExpressionBool = parseOptionExpression(value)

                if (iExpressionBool == null) {
                    SMCLog.severe("Error parsing program condition: $key")
                } else {
                    programConditions[programPath] = iExpressionBool
                }
            }
        }
    }

    private fun parseOptionExpression(name: String): IExpressionBool? {
        return try {
            val resolver = ShaderResolver(this)
            val parser = ExpressionParser(resolver)
            parser.parseBool(name)
        } catch (e: ParseException) {
            SMCLog.warning("${e.javaClass.getName()}: ${e.message}")
            null
        }
    }

    fun getChangedValues() = options.shaderValues.filter { it.value != it.oldValue }.filterIsInstance<IShaderOption>()

    fun applyValues(): Boolean {
        var isChanged = false

        options.shaderValues.forEach {
            if (it.applyValue())
                isChanged = true
        }

        return isChanged
    }

    fun getShaderComponent(value: Value<*>) = when (value) {
        is ShaderListValue -> ShaderOptionListValueComponent(value, this)
        is ShaderBoolValue -> ShaderOptionBoolValueComponent(value, this)
        is ShaderSliderValue -> ShaderOptionSliderValueComponent(value, this)
        is GroupValue -> ShaderOptionGroupValueComponent(value, this)
        else -> null
    }

    open val priority = 2
}