package net.optifine.shaders

import net.bloom.bloomclient.utils.Constants
import net.bloom.bloomclient.value.Value
import net.bloom.bloomclient.value.values.GroupValue
import net.bloom.bloomclient.value.values.shaders.*
import net.minecraft.src.Config
import net.optifine.shaders.config.*
import net.optifine.shaders.file.AbstractShaderPack
import net.optifine.shaders.shaderoption.ShaderProfile
import net.optifine.util.StrUtils
import java.io.*
import java.util.*

object ShaderParserUtils {
    fun getShaderOptionFromLine(line: String, path: String, sliderNames: Set<String>): IShaderOption? {
        val defineBoolMatcher = Constants.PATTERN_DEFINE.matcher(line)

        if (defineBoolMatcher.matches()) {
            val boolean = defineBoolMatcher.group(1)
            val name = defineBoolMatcher.group(2)

            if (name.startsWith("MC_"))
                return null

            val description = defineBoolMatcher.group(3) ?: ""
            if (name != null && name.isNotEmpty())
                return ShaderBoolValue(name, description, boolean != "//", path.removePrefix("/shaders/"), VariableType.DEFINE)
        }

        val defineVariableMatcher = Constants.PATTERN_VARIABLE.matcher(line)

        if (defineVariableMatcher.matches()) {
            val name = defineVariableMatcher.group(1)

            if (name.startsWith("MC_"))
                return null

            val value = defineVariableMatcher.group(2)
            var description = defineVariableMatcher.group(3) ?: ""
            val valuesStr = StrUtils.getSegment(description, "[", "]")

            if (valuesStr != null && valuesStr.isNotEmpty())
                description = description.replace(valuesStr, "").trim { it <= ' ' }

            val values = parseValues(value, valuesStr)
            if (name != null && name.isNotEmpty() && values.isNotEmpty()) {
                return if (name in sliderNames)
                    ShaderSliderValue(name, description, value, values, path.removePrefix("/shaders/"), VariableType.DEFINE)
                else ShaderListValue(name, description, value, values, path.removePrefix("/shaders/"), VariableType.DEFINE)
            }
        }

        val constBoolMatcher = Constants.PATTERN_CONST_BOOL.matcher(line)

        if (constBoolMatcher.matches()) {
            val name = constBoolMatcher.group(1)

            if (name.startsWith("MC_") || name !in Constants.CONST_IFDEF_NAMES)
                return null

            val value = constBoolMatcher.group(2)
            val description = constBoolMatcher.group(3) ?: ""
            if (name != null && name.isNotEmpty()) {
                return ShaderBoolValue(name, description, value.toBoolean(), path.removePrefix("/shaders/"), VariableType.CONST)
            }
        }

        val constVariableMatcher = Constants.PATTERN_CONST_FLOAT_INT.matcher(line)

        if (constVariableMatcher.matches()) {
            val type = constVariableMatcher.group(1)
            val name = constVariableMatcher.group(2)

            if (name.startsWith("MC_") || name !in Constants.CONST_IFDEF_NAMES)
                return null

            val value = constVariableMatcher.group(3)
            var description = constVariableMatcher.group(4) ?: ""
            val valuesStr = StrUtils.getSegment(description, "[", "]")

            if (valuesStr != null && valuesStr.isNotEmpty())
                description = description.replace(valuesStr, "").trim { it <= ' ' }

            val values = parseValues(value, valuesStr)

            if (name != null && name.isNotEmpty() && values.isNotEmpty()) {
                return if (name in sliderNames)
                    ShaderSliderValue(name, description, value, values, path.removePrefix("/shaders/"), VariableType.CONST, type)
                else ShaderListValue(name, description, value, values, path.removePrefix("/shaders/"), VariableType.CONST, type)

            }
        }

        return null
    }

    private fun parseValues(value: String, valuesStr: String?): Array<String> {
        valuesStr ?: return emptyArray()

        val str = valuesStr.trim().removePrefix("[").removeSuffix("]").trim()
        if (str.isEmpty())
            return emptyArray()

        val splitted = str.split(" ").toTypedArray()
        if (splitted.size == 1 && splitted.first() == value)
            return emptyArray()

        return splitted
    }

    @JvmStatic
    fun resolveIncludes(reader: BufferedReader, filePath: String, shaderPack: AbstractShaderPack, fileIndex: Int, listFilePath: MutableSet<String>, includeLevel: Int): BufferedReader {
        var path = "/"
        val endSlashIdx = filePath.lastIndexOf("/")
        if (endSlashIdx >= 0) {
            path = filePath.substring(0, endSlashIdx)
        }

        val writer = CharArrayWriter()
        var idx = -1
        val macros = mutableSetOf<ShaderMacro>()
        var k = 1

        while (true) {
            var line = reader.readLine()
            if (line == null) {
                var chars = writer.toCharArray()
                if (idx >= 0 && macros.isNotEmpty()) {
                    val macroDefines = macros.joinToString("\n") { "#define ${it.name} ${it.value}" }
                    val builder = StringBuilder(String(chars))
                    builder.insert(idx, macroDefines)
                    chars = builder.toString().toCharArray()
                }

                return BufferedReader(CharArrayReader(chars))
            }

            if (idx < 0) {
                val versionMatcher = Constants.PATTERN_VERSION.matcher(line)
                if (versionMatcher.matches()) {
                    val macroLines = ShaderMacros.getFixedMacroLines() + ShaderMacros.getOptionMacroLines()
                    val linesStr = "\n$line\n$macroLines\n"
                    val lineStr = "#line ${k + 1} $fileIndex"
                    line = linesStr + lineStr
                    idx = writer.size() + linesStr.length
                }
            }

            val includePattern = Constants.PATTERN_INCLUDE.matcher(line)
            if (includePattern.matches()) {
                val parsedPath = includePattern.group(1)
                val startsWithSlash = parsedPath.startsWith("/")

                val shaderPath = if (startsWithSlash) "/shaders$parsedPath" else "$path/$parsedPath"
                listFilePath.add(shaderPath)

                val pathIdx = listFilePath.indexOf(shaderPath) + 1
                line = loadFile(shaderPath, shaderPack, pathIdx, listFilePath, includeLevel) ?: throw IOException("Included file not found: $filePath")

                if (line.endsWith("\n"))
                    line = line.substring(0, line.length - 1)

                var firstLine = "#line 1 $pathIdx\n"
                if (line.startsWith("#version "))
                    firstLine = ""

                line = "$firstLine$line\n#line ${k + 1} $fileIndex"
            }

            if (idx >= 0 && line.contains("MC_")) {
                val shaderMacros = ShaderMacros.getExtensions().filter { line.contains(it.name) }.toTypedArray()
                macros.addAll(shaderMacros)
            }

            writer.write(line)
            writer.write("\n")
            ++k
        }
    }

    private fun loadFile(filePath: String, shaderPack: AbstractShaderPack, fileIndex: Int, listFiles: MutableSet<String>, includeLevel: Int): String? {
        if (includeLevel >= 10) {
            throw IOException("#include depth exceeded: $includeLevel, file: $filePath")
        }

        val inputStream = shaderPack.getResourceAsStream(filePath) ?: return null
        var bufferedreader = inputStream.bufferedReader()
        bufferedreader = resolveIncludes(bufferedreader, filePath, shaderPack, fileIndex, listFiles, includeLevel + 1)

        val writer = CharArrayWriter()
        while (true) {
            val line = bufferedreader.readLine() ?: return writer.toString()
            writer.write(line)
            writer.write("\n")
        }
    }

    fun getLines(shaderPack: AbstractShaderPack, path: String): Array<String> {

        try {
            val list = mutableSetOf<String>()
            val s = loadFile(path, shaderPack, 0, list, 0) ?: return emptyArray()
            val inputStream = ByteArrayInputStream(s.toByteArray())
            return inputStream.bufferedReader().use { it.readLines().toTypedArray() }
        } catch (ioexception: IOException) {
            Config.dbg("${ioexception.javaClass.name}: ${ioexception.message}")
        }

        return emptyArray()
    }

    fun parseProfile(shaderPack: AbstractShaderPack, name: String, props: Properties, parsedProfiles: MutableSet<String>): ShaderProfile? {
        if (parsedProfiles.contains("profile.$name")) {
            Config.warn("[Shaders] Profile already parsed: $name")
            return null
        }

        parsedProfiles.add(name)
        val shaderProfile = ShaderProfile(name)
        val s2 = props.getProperty("profile.$name") ?: ""
        val values = s2.split(" ")

        for (value in values) {
            if (value.startsWith("profile.")) {
                val profileName = value.removePrefix("profile")
                val profile = parseProfile(shaderPack, profileName, props, parsedProfiles)
                shaderProfile.addOptionsFromProfile(profile)
                shaderProfile.addDisabledProgramsFromProfile(profile)
                continue
            }

            val property = Config.tokenize(value, "=")
            if (property.size == 1) {
                var key = property[0]
                var isFalse = true
                if (key.startsWith("!")) {
                    isFalse = false
                    key = key.substring(1)
                }

                if (key.startsWith("program.")) {
                    val programKey = key.substring(8)

                    if (!Shaders.isProgramPath(programKey)) {
                        Config.warn("Invalid program: $programKey in profile: ${shaderProfile.name}")
                    } else if (isFalse) {
                        shaderProfile.disabledPrograms.remove(programKey)
                    } else {
                        shaderProfile.disabledPrograms.add(programKey)
                    }

                } else {
                    val shaderOption = shaderPack.options.shaderValues.find { it.name.equals(key, true) }
                    if (shaderOption !is ShaderBoolValue) {
                        Config.warn("[Shaders] Invalid option: $key")
                    } else {
                        shaderProfile.options[key] = isFalse.toString()
                    }
                }

            } else if (property.size != 2) {
                Config.warn("[Shaders] Invalid option value: $value")
            } else {
                val key = property[0]
                val propertyValue = property[1]
                val shaderOption = shaderPack.options.shaderValues.find { it.name.equals(key, true) }

                if (shaderOption is ShaderSliderValue && value !in shaderOption.values) {
                    Config.warn("[Shaders] Invalid value: $propertyValue")
                } else if (shaderOption is ShaderSliderValue && value !in shaderOption.values) {
                    Config.warn("[Shaders] Invalid value: $propertyValue")
                } else if (shaderOption == null || shaderOption is ShaderBoolValue) {
                    Config.warn("[Shaders] Invalid option: $propertyValue")
                } else {
                    shaderProfile.addOption(key, propertyValue)
                }
            }
        }

        return shaderProfile
    }

    fun parseGuiScreen(
        shaderPack: AbstractShaderPack,
        name: String,
        properties: Properties,
    ): GroupValue? {
        val valuesStr = properties.getProperty(name) ?: return null

        val list = mutableListOf<Value<*>>()
        val values = valuesStr.split(" ")

        for (value in values) {
            if (value == "<profile>") {
                shaderPack.options.shaderValues.find { it is ShaderProfileValue }?.let { list.add(it) }
            } else if (value.startsWith("[") && value.endsWith("]")) {
                val screenName = value.removePrefix("[").removeSuffix("]")

                if (!screenName.matches("^[a-zA-Z0-9_]+$".toRegex())) {
                    Config.warn("[Shaders] Invalid screen: $value, key: $name")
                } else {
                    val groupValue = parseGuiScreen(shaderPack, "screen.$screenName", properties)

                    if (groupValue != null) {
                        list.add(groupValue)
                    } else {
                        Config.warn("[Shaders] Invalid screen: $value, key: $name")
                    }
                }
            } else {
                shaderPack.options.shaderValues.find { it.name == value }?.let { list.add(it) }
            }
        }

        return GroupValue(name, name, list.toTypedArray(), { true }, null, false)
    }
}