package net.minecraft.client.option.options.videosettings

import net.bloom.bloomclient.value.values.IntegerValue
import net.minecraft.client.option.MinecraftOption
import net.minecraft.client.option.options.ActionQueue
import net.minecraft.client.option.options.ShadersOption
import net.minecraft.client.renderer.texture.TextureMap
import net.minecraft.client.resources.LanguageManager
import net.minecraft.src.Config
import net.optifine.*

object QualityOption: MinecraftOption("Quality") {

    @JvmStatic var fancyGraphics by boolOption("options.graphics", false) { ActionQueue.isNeedMinecraftLoadRenderers = true }
    @JvmStatic val isSmoothLighting by boolOption("options.ao", false) { ActionQueue.isNeedMinecraftLoadRenderers = true }
    @JvmStatic val smoothLightingLevel by int("options.ao_level", 0, 0, 100)
    private val dynamicLights = listOption("options.dynamic_lights", "options.fast", arrayOf("options.fancy", "options.fast", "options.off")) { DynamicLights.removeLights(mc.renderGlobal) }

    @JvmStatic val mipmapLevels by object: IntegerValue("options.mipmapLevels", 0, 0, 4, this) {
        override val displayedValue: String
            get() = when (temporaryValue) {
                4 -> LanguageManager.format("general.max")
                else -> "${temporaryValue}x"
            }
    }.also {
        it.onPostChange { _, newValue ->
            mc.textureMapBlocks.mipmapLevels = newValue
            mc.textureManager.bindTexture(TextureMap.locationBlocksTexture)
            mc.textureMapBlocks.setBlurMipmapDirect(false, newValue > 0)
            ActionQueue.isNeedMinecraftRefreshResources = true
        }
    }

    private val mipmapType = listOption("options.mipmap_type", "options.mipmap.linear", arrayOf("options.mipmap.linear", "options.mipmap.bilinear", "options.mipmap.trilinear")) {
        ActionQueue.isNeedMinecraftRefreshResources = true
    }

    @JvmStatic val anisotropicFilteringLevel by object: IntegerValue("options.af_level", 0, 0, 4, this) {
        override fun get() = if (Config.isShaders()) 0 else super.get() * 2

        override val displayedValue: String
            get() = when (temporaryValue) {
                1 -> "2x"
                2 -> "4x"
                3 -> "8x"
                4 -> "16x"
                else -> LanguageManager.format("options.off")
            }
    }.also {
        it.onPostChange { _, _ -> ActionQueue.isNeedMinecraftRefreshResources = true }
    }

    @JvmStatic val isCustomFonts by boolOption("options.custom_fonts", true) {
        mc.minecraftFontRendererObj.onResourceManagerReload(Config.getResourceManager())
        mc.standardGalacticMinecraftFontRenderer.onResourceManagerReload(Config.getResourceManager())
    }

    @JvmStatic val isCustomColors by boolOption("options.custom_colors", true) {
        CustomColors.update()
        ActionQueue.isNeedMinecraftLoadRenderers = true
    }

    private val droppedItems = list("options.dropped_items", "options.default", arrayOf("options.default", "options.fast", "options.fancy"))
    private val vignette = list("options.vignette", "options.default", arrayOf("options.default", "options.fast", "options.fancy"))

    @JvmStatic val isSwampColors by boolOption("options.swamp_colors", false) {
        CustomColors.updateUseDefaultGrassFoliageColors()
        ActionQueue.isNeedMinecraftLoadRenderers = true
    }

    @JvmStatic val isSmoothBiomes by boolOption("options.smooth_biomes", false) {
        CustomColors.updateUseDefaultGrassFoliageColors()
        ActionQueue.isNeedMinecraftLoadRenderers = true
    }

    @JvmStatic
    fun isMipmapsEnabled() = mipmapLevels > 0

    @JvmStatic
    fun getMipmapTypeId() = when (mipmapType.get().lowercase()) {
        "options.mipmap.bilinear" -> if (Config.isMultiTexture()) 9985 else 9986
        "options.mipmap.trilinear" -> if (Config.isMultiTexture()) 9987 else 9986
        else -> 9986
    }

    @JvmStatic
    fun isAnisotropicFiltering() = anisotropicFilteringLevel > 1
    @JvmStatic
    fun isDynamicLights() = !dynamicLights.equals("options.off")
    @JvmStatic
    fun isDynamicLightsFast() = dynamicLights.equals("options.fast")
    @JvmStatic
    fun isDynamicHandLight() = isDynamicLights() && (!Config.isShaders() || ShadersOption.isDynamicHandLight())

    @JvmStatic
    fun isDroppedItemsFancy() = if (droppedItems.equals("options.default")) fancyGraphics else droppedItems.equals("options.fancy")

    @JvmStatic
    fun isVignetteEnabled() = (!Config.isShaders() || ShadersOption.isVignette()) && if (vignette.equals("options.default")) fancyGraphics else vignette.equals("options.fancy")


}