package net.minecraft.client.option.options

import net.bloom.bloomclient.value.values.IntegerValue
import net.minecraft.client.option.MinecraftOption
import net.minecraft.client.resources.LanguageManager
import net.minecraft.util.EnumWorldBlockLayer
import net.optifine.shaders.ShaderManager
import net.optifine.shaders.Shaders

object ShadersOption: MinecraftOption("Shaders") {

    @JvmStatic val antialiasingObject = object: IntegerValue("options.shaders.antialiasing", 0, 0, 2, this) {
        override fun get() = value * 2

        override val displayedValue: String
            get() = when (temporaryValue) {
                0 -> LanguageManager.format("options.off")
                else -> "FXAA ${temporaryValue * 2}x"
            }
    }.also {
        it.onPostChange { _, _ -> ActionQueue.isNeedShaderUninit = true }
    }

    @JvmStatic val antialiasing by antialiasingObject

    @JvmStatic val isNormalMap by boolOption("options.shaders.normal_map", false) {
        ActionQueue.isNeedShaderUninit = true
        ActionQueue.isNeedMinecraftScheduledRefreshResources = true
    }

    @JvmStatic val isSpecularMap by boolOption("options.shaders.specular_map", true) {
        ActionQueue.isNeedShaderUninit = true
        ActionQueue.isNeedMinecraftScheduledRefreshResources = true
    }

    @JvmStatic val renderQuality by floatOption("options.shaders.render_res_mul", 1f, 0.25f, 2f) {
        ActionQueue.isNeedShaderUninit = true
        ActionQueue.isNeedScheduleResize = true
    }
    @JvmStatic val shadowQuality by floatOption("options.shaders.shadow_res_mul", 1f, 0.25f, 2f) {
        ActionQueue.isNeedShaderUninit = true
        ActionQueue.isNeedScheduleResizeShadow = true
    }
    @JvmStatic val handDepth by floatOption("options.shaders.hand_depth_mul", 1f, 0.25f, 2f) {
        ActionQueue.isNeedShaderUninit = true
    }

    private val oldLighting = listOption("options.shaders.old_lighting", "options.off", arrayOf("options.default", "options.on", "options.off")) {
        Shaders.updateBlockLightLevel()
        ActionQueue.isNeedShaderUninit = true
        ActionQueue.isNeedMinecraftScheduledRefreshResources = true
    }

    private val oldHandlight = listOption("options.shaders.old_handlight", "options.off", arrayOf("options.default", "options.on", "options.off")) {
        ActionQueue.isNeedShaderUninit = true
    }

    @JvmStatic val isCloudShadow by bool("Cloud shadow", false)

    @JvmStatic val texMinB by createTexMinOption("Tex min B")
    @JvmStatic val texMinN by createTexMinOption("Tex min N")
    @JvmStatic val texMinS by createTexMinOption("Tex min S")

    @JvmStatic val texMagB by createTexMagOption("Tex mag B")
    @JvmStatic val texMagN by createTexMagOption("Tex mag N")
    @JvmStatic val texMagS by createTexMagOption("Tex mag S")

    private fun createTexMinOption(name: String) = object: IntegerValue(name, 0, 0, 2, this) {
        override fun get() = getTexMinNumber(value)

        override val displayedValue: String
            get() = getTexMinNumber(temporaryValue).toString()
    }.also {
        it.onPostChange { _, _ -> ActionQueue.isNeedShaderUpdateMinMag = true }
    }

    private fun createTexMagOption(name: String) = object: IntegerValue(name, 0, 0, 1, this) {
        override fun get() = getTexMagNumber(value)

        override val displayedValue: String
            get() = getTexMagNumber(temporaryValue).toString()
    }.also {
        it.onPostChange { _, _ -> ActionQueue.isNeedShaderUpdateMinMag = true }
    }

    private fun getTexMinNumber(value: Int) = when (value) {
        1 -> 9984
        2 -> 9986
        else -> 9728
    }

    private fun getTexMagNumber(value: Int) = when (value) {
        1 -> 9729
        else -> 9728
    }

    @JvmStatic
    fun isOldLighting() = oldLighting.equals("options.default") && ShaderManager.shaderPack.options.oldLighting || oldLighting.equals("options.on")

    @JvmStatic
    fun isOldHandlight() = oldHandlight.equals("options.default") && ShaderManager.shaderPack.options.oldHandlight || oldHandlight.equals("options.on")

    @JvmStatic
    fun isDynamicHandLight() = ShaderManager.shaderPack.options.dynamicHandlight

    @JvmStatic
    fun isRenderShadowTranslucent() = ShaderManager.shaderPack.options.shadowTranslucent

    @JvmStatic
    fun isUnderwaterOverlay() = ShaderManager.shaderPack.options.underwaterOverlay

    @JvmStatic
    fun isSun() = ShaderManager.shaderPack.options.sun

    @JvmStatic
    fun isMoon() = ShaderManager.shaderPack.options.moon

    @JvmStatic
    fun isVignette() = ShaderManager.shaderPack.options.vignette

    @JvmStatic
    fun isRenderBackFace(blockLayerIn: EnumWorldBlockLayer) = when (blockLayerIn) {
        EnumWorldBlockLayer.SOLID -> ShaderManager.shaderPack.options.backFaceSolid
        EnumWorldBlockLayer.CUTOUT -> ShaderManager.shaderPack.options.backFaceCutout
        EnumWorldBlockLayer.CUTOUT_MIPPED -> ShaderManager.shaderPack.options.backFaceCutoutMipped
        EnumWorldBlockLayer.TRANSLUCENT -> ShaderManager.shaderPack.options.backFaceTranslucent
    }
    @JvmStatic
    fun isRainDepth() = ShaderManager.shaderPack.options.rainDepth
    @JvmStatic
    fun isSeparateAo() = ShaderManager.shaderPack.options.separateAo
    @JvmStatic
    fun isFrustumCulling() = ShaderManager.shaderPack.options.frustumCulling
}