package net.minecraft.client.gui.options

import net.bloom.bloomclient.file.FileManager
import net.bloom.bloomclient.utils.MathUtils
import net.bloom.bloomclient.utils.render.RenderUtils
import net.minecraft.client.gui.GuiScreen
import net.minecraft.client.gui.options.page.OptionPage
import net.minecraft.client.gui.options.page.ResourcePackPage
import net.minecraft.client.gui.options.page.SettingPage
import net.minecraft.client.gui.options.page.ShaderPage
import net.minecraft.client.option.OptionManager
import net.minecraft.client.option.options.ActionQueue
import java.awt.Color
import kotlin.concurrent.thread

class GuiOptions(private val parent: GuiScreen) : GuiScreen() {
    private var currentPage = PageType.GENERAL

    override fun drawScreen(mouseX: Int, mouseY: Int, partialTicks: Float) {
        drawDefaultBackground()

        var xPos = 40

        for (pageType in PageType.entries) {
            val stringWidth = mc.minecraftFontRendererObj.getStringWidth(pageType.pageName)
            val color = if (MathUtils.isHover(mouseX, mouseY, xPos, 15, xPos + stringWidth + 16, 35)) Color(0, 0, 0, 220) else Color(0, 0, 0, 130)
            RenderUtils.drawRect(xPos, 15, xPos + stringWidth + 8, 35, color = color)
            mc.minecraftFontRendererObj.drawString(pageType.pageName, xPos + 4, 25 - mc.minecraftFontRendererObj.FONT_HEIGHT / 2, Color.WHITE)
            xPos += stringWidth + 16
        }

        var xButtonInc = width - 30f

        for (buttonType in ButtonType.entries.reversed()) {
            val stringWidth = mc.minecraftFontRendererObj.getStringWidth(buttonType.buttonName)
            val width = stringWidth + 16f

            xButtonInc -= width

            val color = if (MathUtils.isHover(mouseX, mouseY, xButtonInc, height - 35f, xButtonInc + width, height - 15f))
                Color(0, 0, 0, 220)
            else
                Color(0, 0, 0, 130)

            RenderUtils.drawRect(xButtonInc, height - 35f, xButtonInc + width, height - 15f, color = color)
            mc.minecraftFontRendererObj.drawString(buttonType.buttonName, xButtonInc + 8f, height - 25f - mc.minecraftFontRendererObj.FONT_HEIGHT / 2, Color.WHITE)
            xButtonInc -= 5f
        }

        currentPage.page.drawScreen(mouseX, mouseY, partialTicks)
        super.drawScreen(mouseX, mouseY, partialTicks)
    }


    override fun mouseClicked(mouseX: Int, mouseY: Int, mouseButton: Int) {
        var xButtonInc = width - 30f

        for (buttonType in ButtonType.entries.reversed()) {
            val stringWidth = mc.minecraftFontRendererObj.getStringWidth(buttonType.buttonName)
            val width = stringWidth + 16f

            xButtonInc -= width

            if (MathUtils.isHover(mouseX, mouseY, xButtonInc, height - 35f, xButtonInc + width, height - 15f)) {
                when (buttonType) {
                    ButtonType.APPLY -> applyValue()
                    ButtonType.DONE -> {
                        applyValue()
                        mc.displayGuiScreen(parent)
                    }
                }
                return
            }

            xButtonInc -= 5f
        }

        var xPos = 40

        for (pageType in PageType.entries) {
            val stringWidth = mc.minecraftFontRendererObj.getStringWidth(pageType.pageName)
            if (MathUtils.isHover(mouseX, mouseY, xPos, 15, xPos + stringWidth + 16, 35)) {
                currentPage = pageType
                return
            }

            xPos += stringWidth + 16
        }

        if (currentPage.page.mouseClicked(mouseX, mouseY, mouseButton))
            return
    }

    private fun applyValue() {
        var reloadOnThread = false

        reloadOnThread = reloadOnThread or OptionManager.applyValue()
        reloadOnThread = reloadOnThread or ShaderPage.applyValue()
        reloadOnThread = reloadOnThread or ResourcePackPage.applyResourcePacks()

        FileManager.minecraftConfig.saveConfig()
        FileManager.shaderConfig.saveConfig()
        FileManager.resourceConfig.saveConfig()

        if (reloadOnThread) {
            thread(name = "Reload minecraft thread") { reload() }
        } else {
            reload()
        }
    }

    private fun reload() {
        ActionQueue.applyAction()
        ActionQueue.reset()
    }

    override fun keyTyped(typedChar: Char, keyCode: Int) {
        if (currentPage.page.keyTyped(typedChar, keyCode))
            return

        super.keyTyped(typedChar, keyCode)
    }

    override fun mouseReleased(mouseX: Int, mouseY: Int, state: Int) {
        currentPage.page.mouseReleased(mouseX, mouseY, state)
    }

    override fun mouseClickMove(mouseX: Int, mouseY: Int, clickedMouseButton: Int, timeSinceLastClick: Long) {
        if (clickedMouseButton == 0)
            currentPage.page.mouseClickMove(mouseX, mouseY, clickedMouseButton, timeSinceLastClick)
    }

    enum class PageType(val pageName: String, val page: OptionPage) {
        GENERAL("General", SettingPage(OptionManager.generalCategories)),
        VIDEO_SETTINGS("Video settings", SettingPage(OptionManager.videoSettingsCategories)),
        DEVICES("Devices", SettingPage(OptionManager.devicesCategories)),
        SHADERS("Shaders", ShaderPage),
        RESOURCE_PACKS("Resource packs", ResourcePackPage)
    }

    enum class ButtonType(val buttonName: String) {
        APPLY("Apply"),
        DONE("Done")
    }
}