package net.bloom.bloomclient.value.values

import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonPrimitive
import net.bloom.bloomclient.BloomClient
import net.bloom.bloomclient.features.mode.Mode
import net.bloom.bloomclient.file.Configurable
import net.bloom.bloomclient.value.Value

open class ModeValue<T: Mode>(
    name: String,
    description: String,
    val modes: Array<T>,
    displayable: () -> Boolean,
    owner: Configurable,
    applySettingOnThread: Boolean
): Value<T>(name, description, modes.first(), displayable, owner, applySettingOnThread) {
    init {
        for (mode in modes) {
            mode.values.forEach {
                val valueName = it.name
                it.name = "$name-${mode.name}-$valueName"
            }
        }

        onPreChange {oldValue, newValue ->
            if (owner.state)
                BloomClient.eventManager.unregister(oldValue)

            newValue
        }

        onPostChange {_, newValue ->
            if (owner.state)
                BloomClient.eventManager.register(newValue)
        }
    }

    fun setValue(value: String): Boolean {
        for (mode in modes) {
            if (mode.name.equals(value, true)) {
                set(mode)
                return true
            }
        }

        return false
    }

    override fun toJson(): JsonElement {
        val json = JsonObject()
        json.addProperty("currentMode", value.name)

        val valuesJson = JsonObject()
        val values = modes.flatMap { it.values }
        for (value in values)
            valuesJson.add(value.name, value.toJson())

        json.add("values", valuesJson)
        return json
    }

    override fun fromJson(element: JsonElement) {
        if (!element.isJsonObject)
            return

        val values = modes.flatMap { it.values }

        for ((key, value) in element.asJsonObject.entrySet()) {
            when (key.lowercase()) {
                "currentmode" -> if (value.isJsonPrimitive) setValue(value.asString)
                "values" -> if (value.isJsonObject) {
                    for ((modeKey, modeValue) in value.asJsonObject.entrySet()) {
                        values.find { it.name.equals(modeKey, true) }?.fromJson(modeValue)
                    }
                }
            }
        }
    }
}