package net.bloom.bloomclient.value

import com.google.gson.JsonElement
import net.bloom.bloomclient.file.Configurable
import net.bloom.bloomclient.utils.ClientUtils
import kotlin.reflect.KProperty

abstract class Value<T>(
    var name: String,
    val description: String,
    var value: T,
    var canDisplay: () -> Boolean,
    val owner: Configurable?,
    val applySettingOnThread: Boolean
) {
    var oldValue = value
    val defaultValue = value
    var temporaryValue = value

    private var onPreAction: (oldValue: T, newValue: T) -> T = { _, newValue -> newValue }
    private var onPostAction: (oldValue: T, newValue: T) -> Unit = { _, _ -> }

    init {
        owner?.addValue(this)
    }

    open fun get() = value

    open val displayedValue: String
        get() = temporaryValue.toString()

    fun onPreChange(action: (T, T) -> T) = apply {
        onPreAction = action
    }

    fun onPostChange(action: (T, T) -> Unit) = apply {
        onPostAction = action
    }

    fun set(newValue: T, syncTemporaryValue: Boolean = false): Boolean {
        if (value == newValue)
            return false

        val oldValue = get()

        try {
            val handledValue = onPreAction(oldValue, newValue)
            changeValue(handledValue)
            onPostAction(oldValue, handledValue)
        } catch (e: Exception) {
            ClientUtils.LOGGER.error("Error when change value \"$name\" from \"$oldValue\" to \"$newValue\": ${e.message}: ")
            return false
        } finally {
            temporaryValue = value
        }

        return true
    }

    open fun set(newValue: T) = set(newValue, false)

    fun applyValue() = set(temporaryValue)

    open fun changeValue(newValue: T) {
        oldValue = value
        value = newValue
        syncValue()
    }

    fun syncValue() {
        temporaryValue = value
    }

    operator fun getValue(thisRef: Any?, property: KProperty<*>): T = get()
    operator fun setValue(thisRef: Any?, property: KProperty<*>, value: T) = set(value)

    abstract fun toJson(): JsonElement?
    abstract fun fromJson(element: JsonElement)

    open fun toProperties() = value
    open fun fromProperties(value: Any?) {}
}

enum class RangeSlider {
    LEFT, RIGHT, NONE
}