package net.bloom.bloomclient.utils.render

import net.minecraft.client.MinecraftInstance
import net.minecraft.client.gui.ScaledResolution
import net.minecraft.client.gui.Side
import net.minecraft.client.renderer.GlStateManager
import net.minecraft.client.shader.Framebuffer
import org.lwjgl.nanovg.NVGColor
import org.lwjgl.nanovg.NanoVG
import org.lwjgl.nanovg.NanoVGGL2
import org.lwjgl.opengl.Display
import org.lwjgl.opengl.GL11
import org.lwjgl.opengl.GL11.*
import org.lwjgl.opengl.GL13
import org.lwjgl.opengl.GL20.*
import org.lwjgl.opengl.GL30.*
import java.awt.Color
import java.nio.ByteBuffer


/**
 * A util for NanoVG
 * @author toidicakhia, Blend Client
 */
object NanoVGUtils: MinecraftInstance() {
    val context = NanoVGGL2.nvgCreate(NanoVGGL2.NVG_ANTIALIAS or NanoVGGL2.NVG_STENCIL_STROKES)

    fun createContext(name: String, buffer: ByteBuffer): Long {
        val context = NanoVGUtils.context
        if (context == 0L) {
            throw IllegalStateException("Failed to initialize NanoVG context.")
        }

        val fontId = NanoVG.nvgCreateFontMem(context, name, buffer, false)
        if (fontId == -1) {
            throw IllegalStateException("Failed to load font: $name")
        }

        return context
    }

    fun preRender() {
        GL11.glPushAttrib(1048575)
        GlStateManager.pushMatrix()
        GlStateManager.disableAlpha()
        NanoVG.nvgBeginFrame(context, Display.getWidth().toFloat(), Display.getHeight().toFloat(), 1.0f)
        val sr = ScaledResolution(mc)
        save()
        scale(sr.scaleFactor)
    }

    fun postRender() {
        restore()
        NanoVG.nvgEndFrame(context)
        GlStateManager.enableAlpha()
        GL11.glPopAttrib()
        GlStateManager.popMatrix()
    }

    fun scale(scale: Number) {
        scale(scale, scale)
    }

    fun scale(xScale: Number, yScale: Number) {
        NanoVG.nvgScale(context, xScale.toFloat(), yScale.toFloat())
    }

    fun save() {
        NanoVG.nvgSave(context)
    }

    fun restore() {
        NanoVG.nvgRestore(context)
    }

    fun translate(x: Number, y: Number) {
        NanoVG.nvgTranslate(context, x.toFloat(), y.toFloat())
    }

    fun drawQuads() {
        glBegin(GL_QUADS)
        glTexCoord2f(0f, 0f)
        glVertex2f(-1f, -1f)
        glTexCoord2f(0f, 1f)
        glVertex2f(-1f, 1f)
        glTexCoord2f(1f, 1f)
        glVertex2f(1f, 1f)
        glTexCoord2f(1f, 0f)
        glVertex2f(1f, -1f)
        glEnd()
    }

    fun bindTexture(texture: Int) {
        GL13.glActiveTexture(GL13.GL_TEXTURE0)
        glBindTexture(GL_TEXTURE_2D, texture)
    }

    fun renderScissorBox(x: Number, y: Number, width: Number, height: Number, renderFunc: () -> Unit) {
        NanoVG.nvgScissor(context, x.toFloat(), y.toFloat(), width.toFloat(), height.toFloat())
        renderFunc()
        NanoVG.nvgResetScissor(context)
    }

    /**
     * Draws a filled circle at the specified position with the given radius and color.
     *
     * @param centerX X coordinate of the circle's center.
     * @param centerY Y coordinate of the circle's center.
     * @param radius Radius of the circle.
     * @param color Color of the circle.
     */
    fun drawCircle(centerX: Number, centerY: Number, radius: Number, color: Color) {
        val nvgColor = NVGColor.calloc()
        try {
            NanoVG.nvgBeginPath(context)
            NanoVG.nvgShapeAntiAlias(context, true)
            NanoVG.nvgCircle(context, centerX.toFloat(), centerY.toFloat(), radius.toFloat())
            NanoVG.nvgRGBAf(
                color.red.toFloat() / 255f,
                color.green.toFloat() / 255f,
                color.blue.toFloat() / 255f,
                color.alpha.toFloat() / 255f,
                nvgColor
            )
            NanoVG.nvgFillColor(context, nvgColor)
            NanoVG.nvgFill(context)
            NanoVG.nvgClosePath(context)
        } catch (e: Throwable) {
            e.printStackTrace()
        } finally {
            nvgColor.close()
        }
    }

    fun drawRoundedRect(startX: Float, startY: Float, endX: Float, endY: Float, radius: Float, color: Color) {
        drawRect(
            startX, startY, endX, endY, radius, color,
            roundTL = true, roundTR = true, roundBL = true, roundBR = true
        )
    }

    fun drawBorderRect(startX: Number, startY: Number, endX: Number, endY: Number, color: Color = Color.WHITE, borderWidth: Float = 1f) {
        drawRect(
            startX.toFloat(), startY.toFloat(), endX.toFloat(), endY.toFloat(), 0f, color,
            roundTL = false,
            roundTR = false,
            roundBL = false,
            roundBR = false,
            border = true,
            borderWidth = borderWidth,
        )
    }

    fun drawBorderRect(startX: Float, startY: Float, endX: Float, endY: Float, color: Color = Color.WHITE, borderWidth: Float = 1f) {
        drawRect(
            startX, startY, endX, endY, 0f, color,
            roundTL = false,
            roundTR = false,
            roundBL = false,
            roundBR = false,
            border = true,
            borderWidth = borderWidth,
        )
    }


    fun drawRect(startXIn: Int, startYIn: Int, endXIn: Int, endYIn: Int, color: Color) {
        drawRect(startXIn.toFloat(), startYIn.toFloat(), endXIn.toFloat(), endYIn.toFloat(), color = color)
    }

    @JvmOverloads
    fun drawRect(
        startXIn: Float, startYIn: Float, endXIn: Float, endYIn: Float,
        radius: Float = 0f, color: Color = Color.WHITE,
        roundTL: Boolean = false, roundTR: Boolean = false,
        roundBL: Boolean = false, roundBR: Boolean = false,
        border: Boolean = false, borderWidth: Float = 1f) {
        val nvgColor = NVGColor.calloc()

        val radTL = if (roundTL) radius else 0f
        val radTR = if (roundTR) radius else 0f
        val radBR = if (roundBR) radius else 0f
        val radBL = if (roundBL) radius else 0f

        NanoVG.nvgBeginPath(context)
        NanoVG.nvgShapeAntiAlias(context, true)
        NanoVG.nvgRoundedRectVarying(context, startXIn, startYIn, endXIn, endYIn, radTL, radTR, radBR, radBL)
        NanoVG.nvgRGBAf(color.red / 255f, color.green / 255f, color.blue / 255f, color.alpha / 255f, nvgColor)

        if (border) {
            NanoVG.nvgStrokeWidth(context, borderWidth)
            NanoVG.nvgStrokeColor(context, nvgColor)
            NanoVG.nvgStroke(context)
        } else {
            NanoVG.nvgFillColor(context, nvgColor)
            NanoVG.nvgFill(context)
        }

        NanoVG.nvgClosePath(context)

        nvgColor.close()
    }

    fun drawOutlineRoundedRect(
        x: Number,
        y: Number,
        width: Number,
        height: Number,
        stroke: Number,
        radius: Number,
        fillColor: Color,
        borderColor: Color
    ) {
        val nvgColorFill = NVGColor.calloc()
        val nvgColorStroke = NVGColor.calloc()

        try {
            NanoVG.nvgBeginPath(context)
            NanoVG.nvgShapeAntiAlias(context, true)
            NanoVG.nvgRoundedRect(
                context,
                x.toFloat(),
                y.toFloat(),
                width.toFloat(),
                height.toFloat(),
                radius.toFloat()
            )

            NanoVG.nvgRGBAf(
                fillColor.red.toFloat() / 255f,
                fillColor.green.toFloat() / 255f,
                fillColor.blue.toFloat() / 255f,
                fillColor.alpha.toFloat() / 255f,
                nvgColorFill
            )

            NanoVG.nvgRGBAf(
                borderColor.red.toFloat() / 255f,
                borderColor.green.toFloat() / 255f,
                borderColor.blue.toFloat() / 255f,
                borderColor.alpha.toFloat() / 255f,
                nvgColorStroke
            )

            NanoVG.nvgFillColor(context, nvgColorFill)
            NanoVG.nvgFill(context)

            NanoVG.nvgStrokeWidth(context, stroke.toFloat())
            NanoVG.nvgStrokeColor(context, nvgColorStroke)
            NanoVG.nvgStroke(context)

            NanoVG.nvgClosePath(context)
        } catch (e: Throwable) {
            e.printStackTrace()
        } finally {
            nvgColorFill.close()
            nvgColorStroke.close()
        }
    }

    /**
     * @author: longathelstan
     */
    fun drawShadowRoundedRect(
        x: Float, y: Float,
        width: Float, height: Float,
        radius: Float, shadowSize: Float,
        shadowAlpha: Float, color: Color
    ) {
        for (i in 1..5) {
            val alpha = (shadowAlpha * (1f - i / 6f) * 255).toInt()
            val shadowColor = Color(0, 0, 0, alpha)
            val offset = shadowSize * i / 5f

            drawRect(
                x - offset, y - offset,
                width + offset * 2, height + offset * 2,
                radius + offset / 2f, shadowColor
            )
        }

        drawRect(x, y, width, height, radius, color)
    }

    fun drawShadowOutlineRoundedRect(
        x: Float, y: Float,
        width: Float, height: Float,
        stroke: Float, radius: Float,
        shadowSize: Float, shadowAlpha: Float,
        fillColor: Color, borderColor: Color
    ) {
        for (i in 1..5) {
            val alpha = (shadowAlpha * (1f - i / 6f) * 255).toInt()
            val shadowColor = Color(0, 0, 0, alpha)
            val offset = shadowSize * i / 5f

            drawRect(
                x - offset, y - offset,
                width + offset * 2, height + offset * 2,
                radius + offset / 2f, shadowColor
            )
        }

        drawOutlineRoundedRect(x, y, width, height, stroke, radius, fillColor, borderColor)
    }

    fun getNanoVGSide(side: Side): Int {
        val horizontalAlign = when (side.horizontal) {
            Side.Horizontal.LEFT -> NanoVG.NVG_ALIGN_LEFT
            Side.Horizontal.MIDDLE -> NanoVG.NVG_ALIGN_CENTER
            Side.Horizontal.RIGHT -> NanoVG.NVG_ALIGN_RIGHT
        }

        val verticalAlign = when (side.vertical) {
            Side.Vertical.TOP -> NanoVG.NVG_ALIGN_TOP
            Side.Vertical.MIDDLE -> NanoVG.NVG_ALIGN_MIDDLE
            Side.Vertical.BOTTOM -> NanoVG.NVG_ALIGN_BOTTOM
        }

        return horizontalAlign or verticalAlign
    }

    fun drawNanoVG(drawFunction: () -> Unit) {
        preRender()
        save()

        drawFunction()

        restore()
        postRender()
    }
}