package net.bloom.bloomclient.utils.humanization
import org.json.JSONObject
import java.io.File
import kotlin.math.abs
import kotlin.math.sqrt
import kotlin.random.Random
import kotlin.math.ln

class ClickIntervalModelWrapper(modelPath: String) {
    private val weights: DoubleArray
    private val means: DoubleArray
    private val covariances: DoubleArray
    private val random = Random(System.currentTimeMillis())

    init {
        val json = JSONObject(File(modelPath).readText())
        weights = json.getJSONArray("weights").let { arr -> DoubleArray(arr.length()) { arr.getDouble(it) } }
        means = json.getJSONArray("means").let { arr -> DoubleArray(arr.length()) { arr.getDouble(it) } }
        covariances = json.getJSONArray("covariances").let { arr -> DoubleArray(arr.length()) { arr.getDouble(it) } }
    }
    private fun sampleComponent(): Int {
        val r = random.nextDouble()
        var cum = 0.0 // cumulative sum, dont be dirty
        for (i in weights.indices) {
            cum += weights[i]
            if (r <= cum) return i
        }
        return weights.lastIndex
    }
    fun sampleInterval(jitter: Double = 0.02): Double {
        val comp = sampleComponent()
        val mean = means[comp]
        val std = sqrt(covariances[comp])
        val base = mean + std * random.nextGaussian()
        val noisy = base + base * (random.nextGaussian() * jitter)
        return abs(noisy).coerceAtLeast(1.0)
    }

    /** Generate N intervals */
    fun generate(n: Int, jitter: Double = 0.02): List<Double> {
        return List(n) { sampleInterval(jitter) }
    }

    fun Random.nextGaussian(): Double {
        var u: Double
        var v: Double
        var s: Double
        do {
            u = nextDouble() * 2 - 1
            v = nextDouble() * 2 - 1
            s = u * u + v * v
        } while (s >= 1 || s == 0.0)
        val mul = sqrt(-2.0 * ln(s) / s)
        return u * mul
    }
}
