package net.bloom.bloomclient.ui.hud.element.elements.arraylist

import net.bloom.bloomclient.BloomClient
import net.bloom.bloomclient.ui.hud.element.Border
import net.bloom.bloomclient.ui.hud.element.elements.ArrayList
import net.bloom.bloomclient.utils.render.skiko.Skiko
import net.bloom.bloomclient.utils.render.skiko.SkikoContext
import net.bloom.bloomclient.utils.render.skiko.font.Fonts
import net.minecraft.client.gui.Side
import org.jetbrains.skia.ImageFilter
import org.jetbrains.skia.Paint
import org.jetbrains.skia.Path
import org.jetbrains.skia.PathEffect
import org.jetbrains.skia.PathOp
import org.jetbrains.skia.Rect
import java.awt.Color
import kotlin.math.max

class BloomArrayListStyle(parent: ArrayList) : ArrayListStyle(parent, "Bloom") {
    private val nunito = Fonts.NUNITO[10F]
    private val metrics = nunito.metrics

    private val backgroundColor = Color(32, 34, 37, 220)
    private val textColor = Color(255, 255, 255)

    override fun drawElement(): Border {
        val sortedModules = BloomClient.moduleManager.modules
            .filter { it.state || it.animation.value > 0.01f } // Show if enabled or still fading
            .sortedByDescending { nunito.width(it.displayName) }

        if (sortedModules.isEmpty()) return Border(0f, 0f, 0f, 0f)

        updateAnimations()

        var yPos = parent.renderY
        var maxWidth = 0f
        var maxHeight = 0f

        val padding = 3f
        val lineSpacing = 0f
        val lineHeight = metrics.descent - metrics.ascent

        SkikoContext.preFlush()

        var bgPath = Path()
        bgPath.use {
            var tempY = yPos

            for (module in sortedModules) {
                val animationValue = module.animation.value
                val textWidth = nunito.width(module.displayName)
                val animatedHeight = lineHeight * animationValue

                // Calculate position based on side settings and animation
                val xOffset = when (parent.side.horizontal) {
                    Side.Horizontal.LEFT -> 0f
                    Side.Horizontal.MIDDLE -> -textWidth / 2
                    Side.Horizontal.RIGHT -> -textWidth
                }

                val xPos = parent.renderX + xOffset

                // Apply slide animation
                val animatedXPos = when (parent.side.horizontal) {
                    Side.Horizontal.LEFT -> xPos - (1f - animationValue) * textWidth
                    Side.Horizontal.MIDDLE -> xPos
                    Side.Horizontal.RIGHT -> xPos + (1f - animationValue) * textWidth
                }

                val bounds = Rect.makeLTRB(
                    animatedXPos - padding,
                    tempY,
                    animatedXPos + textWidth + padding,
                    tempY + animatedHeight
                )

                bgPath = Path.makeCombining(bgPath, Path().addRect(bounds), PathOp.UNION) ?: Path()
                tempY += animatedHeight
            }

            val overallAlpha = sortedModules.maxOf { it.animation.value }

            val animatedBgPaint = Paint().apply {
                color = Color(0, 0, 0, (150 * overallAlpha).toInt()).rgb
                imageFilter = ImageFilter.makeDropShadow(0f, 0f, 10f, 10f, color)
                pathEffect = PathEffect.makeCorner(5f)
            }

            Skiko.drawPath(bgPath, animatedBgPaint)

            var textY = yPos + metrics.height / 2

            for (module in sortedModules) {
                val animationValue = module.animation.value
                val textWidth = nunito.width(module.displayName)
                val textHeight = lineHeight + padding * 2

                // Calculate position based on side settings and animation
                val xOffset = when (parent.side.horizontal) {
                    Side.Horizontal.LEFT -> 0f
                    Side.Horizontal.MIDDLE -> -textWidth / 2
                    Side.Horizontal.RIGHT -> -textWidth
                }

                val xPos = parent.renderX + xOffset

                // Apply slide animation
                val animatedXPos = when (parent.side.horizontal) {
                    Side.Horizontal.LEFT -> xPos - (1f - animationValue) * textWidth
                    Side.Horizontal.MIDDLE -> xPos
                    Side.Horizontal.RIGHT -> xPos + (1f - animationValue) * textWidth
                }

                val animatedTextPaint = Paint().apply {
                    color = Color(
                        (255 * animationValue).toInt(),
                        (255 * animationValue).toInt(),
                        (255 * animationValue).toInt(),
                        (255 * animationValue).toInt()
                    ).rgb
                }

                nunito.text(
                    module.displayName,
                    animatedXPos,
                    textY + metrics.descent,
                    animatedTextPaint
                )

                textY += (lineHeight * animationValue) + lineSpacing
                yPos += lineHeight * animationValue

                maxWidth = max(maxWidth, textWidth)
                maxHeight += textHeight * animationValue
            }
        }

        SkikoContext.flush()

        // Calculate border based on alignment
        return when (parent.side.horizontal) {
            Side.Horizontal.LEFT -> Border(
                parent.renderX,
                parent.renderY,
                parent.renderX + maxWidth,
                parent.renderY + maxHeight
            )
            Side.Horizontal.MIDDLE -> Border(
                parent.renderX - maxWidth / 2,
                parent.renderY,
                parent.renderX + maxWidth / 2,
                parent.renderY + maxHeight
            )
            Side.Horizontal.RIGHT -> Border(
                parent.renderX - maxWidth,
                parent.renderY,
                parent.renderX,
                parent.renderY + maxHeight
            )
        }
    }

    /**
     * Update animation values for each module
     */
    private fun updateAnimations() {
        BloomClient.moduleManager.modules.forEach {
            it.animation.run(if (it.state) 1f else 0f)
        }
    }
}