package net.bloom.bloomclient.file

import com.google.gson.JsonNull
import com.google.gson.JsonObject
import com.google.gson.JsonParser
import net.bloom.bloomclient.BloomClient
import net.bloom.bloomclient.file.configs.*
import net.bloom.bloomclient.utils.ClientUtils
import net.bloom.bloomclient.utils.io.FileUtils
import net.minecraft.client.MinecraftInstance
import java.io.BufferedReader
import java.io.File
import java.io.FileReader


object FileManager: MinecraftInstance() {
	// Folder
	var dir = FileUtils.makeFolder(mc.mcDataDir, BloomClient.CLIENT_FOLDER)
	val fontsFolder = FileUtils.makeFolder(dir, "fonts")
	val cacheFolder = FileUtils.makeFolder(dir, "cache")
	val imagesFolder = FileUtils.makeFolder(dir, "images")
	val configFolder = FileUtils.makeFolder(dir, "configs")

	// Config
	val moduleConfig = ModuleConfig(File(dir, "modules.json"))
	val accountsConfig = AccountsConfig(File(dir, "accounts.json"))
	val minecraftConfig = MinecraftConfig(File(mc.mcDataDir, "settings.json"))
	val shaderConfig = ShaderConfig(File(mc.mcDataDir, "shaderSettings.json"))
	val resourceConfig = ResourceConfig(File(mc.mcDataDir, "resourceSettings.json"))

	fun loadAllConfigs() {
		javaClass.declaredFields.filter { it.type == FileConfig::class.java }.forEach {
			try {
				it.isAccessible = true
				loadConfig(it[this] as FileConfig)
			} catch(e: IllegalAccessException) {
                ClientUtils.LOGGER.error("[FileManager] Failed to load config file of field ${it.name}.", e)
            }
		}
	}

	fun loadConfig(config: FileConfig) {
        if (!config.hasConfig()) {
            ClientUtils.LOGGER.info("[FileManager] Skipped loading config: ${config.file.name}.")
            saveConfig(config, true)
            return
        }

        try {
            config.loadConfig()
            ClientUtils.LOGGER.info("[FileManager] Loaded config: ${config.file.name}.")
        } catch(t: Throwable) {
            ClientUtils.LOGGER.error("[FileManager] Failed to load config file: ${config.file.name}.", t)
        }
    }

    fun saveAllConfigs() {
		javaClass.declaredFields.filter { it.type == FileConfig::class.java }.forEach {
			try {
				it.isAccessible = true
				saveConfig(it[this] as FileConfig)
			} catch(e: IllegalAccessException) {
                ClientUtils.LOGGER.error("[FileManager] Failed to load config file of field ${it.name}.", e)
            }
		}
	}

    fun saveConfig(config: FileConfig, ignoreStarting: Boolean = false) {
        if (!ignoreStarting && BloomClient.isStarting) return

        try {
            if (!config.hasConfig())
            	config.createConfig()
            
            config.saveConfig()
            ClientUtils.LOGGER.info("[FileManager] Saved config: ${config.file.name}.")
        } catch(t: Throwable) {
            ClientUtils.LOGGER.error("[FileManager] Failed to save config file: ${config.file.name}.", t)
        }
    }

    fun loadConfigFromFile(file: File) {
    	val json = JsonParser.parseReader(BufferedReader(FileReader(file)))

    	if (json is JsonNull || !json.isJsonObject)
    		return

    	json.asJsonObject["modules"]?.asJsonObject?.let {
    		moduleConfig.fromJson(it)
    		moduleConfig.saveConfig()
    	}
    }

    fun saveConfigToFile(file: File) {
    	val json = JsonObject()
    	json.add("modules", moduleConfig.toJson())

    	FileUtils.writeJson(file, json)
    }

}